package ru.yandex.direct.mysql.schema;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.regex.Pattern;

import javax.annotation.Nullable;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonGetter;
import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonProperty;

import ru.yandex.direct.mysql.MySQLUtils;

public class TableSchema {
    private static final Pattern BAD_JSON_DEFAULT = Pattern.compile("json NOT NULL DEFAULT 'null'");
    private static final String BAD_JSON_REPLACEMENT = "json NOT NULL";

    private final String name;
    private final TableType type;
    private final String createSql;
    private final List<ColumnSchema> columns;
    private final List<KeySchema> keys;
    private final List<TriggerSchema> triggers;

    @JsonCreator
    public TableSchema(@JsonProperty("name") String name, @JsonProperty("type") TableType type,
                       @JsonProperty("create_sql") String createSql, @JsonProperty("columns") List<ColumnSchema> columns,
                       @JsonProperty("keys") List<KeySchema> keys, @JsonProperty("triggers") List<TriggerSchema> triggers) {
        this.name = Objects.requireNonNull(name);
        this.type = Objects.requireNonNull(type);
        this.createSql = Objects.requireNonNull(createSql);
        this.columns = Objects.requireNonNull(columns);
        this.keys = Objects.requireNonNull(keys);
        this.triggers = Objects.requireNonNull(triggers);
    }

    @JsonGetter("name")
    public String getName() {
        return name;
    }

    @JsonGetter("type")
    public TableType getType() {
        return type;
    }

    @JsonGetter("create_sql")
    public String getCreateSql() {
        return createSql;
    }

    @JsonGetter("columns")
    public List<ColumnSchema> getColumns() {
        return columns;
    }

    @JsonGetter("keys")
    public List<KeySchema> getKeys() {
        return keys;
    }

    @JsonIgnore
    public Optional<KeySchema> getPrimaryKey() {
        for (KeySchema key : keys) {
            if (key.isPrimary()) {
                return Optional.of(key);
            }
        }
        return Optional.empty();
    }

    @JsonIgnore
    public Optional<KeySchema> getSoleUniqueKey() {
        Optional<KeySchema> result = Optional.empty();
        for (KeySchema key : keys) {
            if (key.isUnique()) {
                if (result.isPresent()) {
                    // В таблице больше одного уникального ключа
                    return Optional.empty();
                } else {
                    result = Optional.of(key);
                }
            }
        }
        return result;
    }

    @JsonGetter("triggers")
    public List<TriggerSchema> getTriggers() {
        return triggers;
    }

    public static TableSchema dump(Connection conn, String tableName) throws SQLException {
        TableType tableType = TableType.TABLE;
        String createSql = null;
        try (PreparedStatement stmt = conn.prepareStatement("SHOW CREATE TABLE " + MySQLUtils.quoteName(tableName))) {
            try (ResultSet rs = stmt.executeQuery()) {
                ResultSetMetaData metaData = rs.getMetaData();
                if ("View".equals(metaData.getColumnName(1))) {
                    tableType = TableType.VIEW;
                }
                if (rs.next()) {
                    createSql = rs.getString(2);
                }
            }
        }
        if (createSql == null) {
            throw new IllegalStateException("Cannot get sql for create table statemement");
        }
        String databaseName = conn.getCatalog();
        List<ColumnSchema> columns = new ArrayList<>();
        Map<String, ColumnSchema> columnsByName = new HashMap<>();
        try (PreparedStatement stmt = conn.prepareStatement(
                "SELECT COLUMN_NAME, DATA_TYPE, COLUMN_TYPE, COLUMN_DEFAULT, IS_NULLABLE from information_schema.COLUMNS where TABLE_SCHEMA = ? and TABLE_NAME = ? order by ORDINAL_POSITION")) {
            stmt.setString(1, databaseName);
            stmt.setString(2, tableName);
            try (ResultSet rs = stmt.executeQuery()) {
                while (rs.next()) {
                    ColumnSchema column =
                            new ColumnSchema(rs.getString(1), rs.getString(2), rs.getString(3), rs.getString(4),
                                    "YES".equalsIgnoreCase(rs.getString(5)));
                    columns.add(column);
                    columnsByName.put(column.getName(), column);
                }
            }
        }
        List<KeySchema> keys = new ArrayList<>();
        Map<String, KeySchema> keysByName = new HashMap<>();
        try (PreparedStatement stmt = conn.prepareStatement("SHOW KEYS IN " + MySQLUtils.quoteName(tableName))) {
            try (ResultSet rs = stmt.executeQuery()) {
                while (rs.next()) {
                    // Result columns: http://dev.mysql.com/doc/refman/5.7/en/show-index.html
                    String keyName = rs.getString("Key_name");
                    KeySchema key = keysByName.get(keyName);
                    if (key == null) {
                        key = new KeySchema(keyName, rs.getString("Index_type"), !rs.getBoolean("Non_unique"),
                                new ArrayList<>());
                        keys.add(key);
                        keysByName.put(keyName, key);
                    }
                    int columnIndex = rs.getInt("Seq_in_index") - 1;
                    if (columnIndex < 0) {
                        throw new IllegalStateException("Found key column with Seq_in_index=" + (columnIndex + 1));
                    }
                    String columnName = rs.getString("Column_name");
                    Integer columnPart = rs.getInt("Sub_part");
                    if (rs.wasNull()) {
                        columnPart = null;
                    }
                    boolean isNullable = "YES".equalsIgnoreCase(rs.getString("Null"));
                    KeyColumn column = new KeyColumn(columnName, columnPart, isNullable);
                    List<KeyColumn> keyColumns = key.getColumns();
                    while (keyColumns.size() < columnIndex) {
                        keyColumns.add(null);
                    }
                    if (keyColumns.size() == columnIndex) {
                        keyColumns.add(column);
                    } else {
                        keyColumns.set(columnIndex, column);
                    }
                }
            }
        }
        List<String> triggerNames = new ArrayList<>();
        try (PreparedStatement stmt = conn
                .prepareStatement("SHOW TRIGGERS LIKE " + MySQLUtils.quoteLike(tableName))) {
            try (ResultSet rs = stmt.executeQuery()) {
                while (rs.next()) {
                    triggerNames.add(rs.getString(1));
                }
            }
        }
        List<TriggerSchema> triggers = new ArrayList<>();
        for (String triggerName : triggerNames) {
            triggers.add(TriggerSchema.dump(conn, triggerName));
        }
        return new TableSchema(tableName, tableType, createSql, columns, keys, triggers);
    }

    public void restore(Connection conn) throws SQLException {
        String sql;
        switch (type) {
            case TABLE:
                sql = createSql;
                break;
            case VIEW:
                // fake preliminary view for resolving interdependencies
                StringBuilder sb = new StringBuilder();
                sb.append("CREATE VIEW ");
                sb.append(MySQLUtils.quoteName(name));
                sb.append(" AS SELECT ");
                boolean first = true;
                for (ColumnSchema column : columns) {
                    if (first) {
                        first = false;
                    } else {
                        sb.append(", ");
                    }
                    sb.append("1 AS ");
                    sb.append(MySQLUtils.quoteName(column.getName()));
                }
                sql = sb.toString();
                break;
            default:
                throw new IllegalStateException("Unsupported table type");
        }
        // DIRECT-70377: успешно применённый ALTER с кривым DEFAULT выдаёт кривую схему на SHOW CREATE TABLE
        sql = BAD_JSON_DEFAULT.matcher(sql).replaceAll(BAD_JSON_REPLACEMENT);
        MySQLUtils.executeUpdate(conn, sql);
        for (TriggerSchema trigger : triggers) {
            trigger.restore(conn);
        }
    }

    public void restoreViews(Connection conn) throws SQLException {
        if (type == TableType.VIEW) {
            // drop the fake view
            MySQLUtils.executeUpdate(conn, "DROP VIEW IF EXISTS " + MySQLUtils.quoteName(name));
            MySQLUtils.executeUpdate(conn, createSql);
        }
    }

    public boolean schemaEquals(TableSchema that) {
        if (!name.equals(that.name)) {
            return false;
        }
        if (type != that.type) {
            return false;
        }
        if (!columns.equals(that.columns)) {
            return false;
        }
        if (!keys.equals(that.keys)) {
            return false;
        }
        return triggers.equals(that.triggers);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof TableSchema)) {
            return false;
        }

        TableSchema that = (TableSchema) o;

        if (!name.equals(that.name)) {
            return false;
        }
        if (type != that.type) {
            return false;
        }
        if (!createSql.equals(that.createSql)) {
            return false;
        }
        if (!columns.equals(that.columns)) {
            return false;
        }
        if (!keys.equals(that.keys)) {
            return false;
        }
        return triggers.equals(that.triggers);

    }

    @Nullable
    public ColumnSchema findColumn(String name) {
        return columns.stream().filter(col -> col.getName().equals(name)).findFirst().orElse(null);
    }

    @Override
    public int hashCode() {
        int result = name.hashCode();
        result = 31 * result + type.hashCode();
        result = 31 * result + createSql.hashCode();
        result = 31 * result + columns.hashCode();
        result = 31 * result + keys.hashCode();
        result = 31 * result + triggers.hashCode();
        return result;
    }
}
