package ru.yandex.direct.mysql.schema;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Objects;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonGetter;
import com.fasterxml.jackson.annotation.JsonProperty;

import ru.yandex.direct.mysql.MySQLUtils;

public class TriggerSchema {
    private final String name;
    private final String createSql;

    @JsonCreator
    public TriggerSchema(@JsonProperty("name") String name, @JsonProperty("create_sql") String createSql) {
        this.name = Objects.requireNonNull(name);
        this.createSql = Objects.requireNonNull(createSql);
    }

    @JsonGetter("name")
    public String getName() {
        return name;
    }

    @JsonGetter("create_sql")
    public String getCreateSql() {
        return createSql;
    }

    public static TriggerSchema dump(Connection conn, String triggerName) throws SQLException {
        String createSql = null;
        try (PreparedStatement stmt = conn
                .prepareStatement("SHOW CREATE TRIGGER " + MySQLUtils.quoteName(triggerName))) {
            try (ResultSet rs = stmt.executeQuery()) {
                if (rs.next()) {
                    createSql = rs.getString(3);
                }
            }
        }
        if (createSql == null) {
            throw new IllegalStateException("Failed to get create trigger statement for " + triggerName);
        }
        return new TriggerSchema(triggerName, createSql);
    }

    public void restore(Connection conn) throws SQLException {
        try (PreparedStatement stmt = conn.prepareStatement(createSql)) {
            stmt.executeUpdate();
        }
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof TriggerSchema)) {
            return false;
        }

        TriggerSchema that = (TriggerSchema) o;

        if (!name.equals(that.name)) {
            return false;
        }
        return createSql.equals(that.createSql);

    }

    @Override
    public int hashCode() {
        int result = name.hashCode();
        result = 31 * result + createSql.hashCode();
        return result;
    }
}
