package ru.yandex.direct.mysql.util;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.math.BigDecimal;
import java.nio.charset.StandardCharsets;

import com.fasterxml.jackson.core.JsonEncoding;
import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonToken;
import com.fasterxml.jackson.core.util.MinimalPrettyPrinter;
import com.google.common.base.Preconditions;

/**
 * Реализует pretty-print, соответствующий тому, который использует mysql
 * Строки, полученные таким образом, должны точно соответствовать тому, что возвращает mysql
 * MySQL also discards extra whitespace between keys, values, or elements in the original JSON
 * document, and leaves (or inserts, when necessary) a single space following each comma (,)
 * or colon (:) when displaying it. This is done to enhance readibility.
 * https://dev.mysql.com/doc/refman/8.0/en/json.html#json-normalization
 * <p>
 * + workaround для бага https://bugs.mysql.com/bug.php?id=88230
 * для этого json перекладывается целиком, при этом double-значения с trailingZeros заменяются
 * на значения без trailingZeros (так же, как это делает MySQL до версии 8.04)
 */
public class Mysql57CompatibleJsonFormatter {
    /**
     * MySQL форматирует JSON именно таким образом
     */
    public static class MysqlPrettyPrinter extends MinimalPrettyPrinter {
        @Override
        public void writeObjectFieldValueSeparator(JsonGenerator g) throws IOException {
            super.writeObjectFieldValueSeparator(g);
            g.writeRaw(" ");
        }

        @Override
        public void writeArrayValueSeparator(JsonGenerator g) throws IOException {
            super.writeArrayValueSeparator(g);
            g.writeRaw(" ");
        }

        @Override
        public void writeObjectEntrySeparator(JsonGenerator g) throws IOException {
            super.writeObjectEntrySeparator(g);
            g.writeRaw(" ");
        }
    }

    public static class Mysql57CompatibleJsonFormatException extends RuntimeException {
        Mysql57CompatibleJsonFormatException(String message) {
            super(message);
        }

        Mysql57CompatibleJsonFormatException(String message, Throwable cause) {
            super(message, cause);
        }
    }

    private static final MysqlPrettyPrinter PRETTY_PRINTER = new MysqlPrettyPrinter();

    public static String format(String json) throws Mysql57CompatibleJsonFormatException {
        JsonFactory jsonFactory = new JsonFactory();
        try (JsonParser parser = jsonFactory.createParser(json)) {
            try (ByteArrayOutputStream ostream = new ByteArrayOutputStream()) {
                try (JsonGenerator generator = jsonFactory.createGenerator(ostream, JsonEncoding.UTF8)) {
                    generator.setPrettyPrinter(PRETTY_PRINTER);
                    //
                    processRootValue(parser, generator);
                }
                return ostream.toString(StandardCharsets.UTF_8);
            }
        } catch (IOException e) {
            throw new Mysql57CompatibleJsonFormatException("Unexpected exception while formatting JSON", e);
        }
    }

    private static void processRootValue(JsonParser jParser, JsonGenerator jGenerator) throws IOException {
        jParser.nextToken();
        processValue(jParser, jGenerator);
    }

    private static void processValue(JsonParser jParser, JsonGenerator jGenerator) throws IOException {
        JsonToken token = jParser.currentToken();
        if (null == token) {
            throw new Mysql57CompatibleJsonFormatException("Unexpected EOF");
        }
        switch (token) {
            case VALUE_STRING:
                jGenerator.writeString(jParser.getText());
                break;
            case VALUE_NUMBER_INT:
            case VALUE_TRUE:
            case VALUE_FALSE:
            case VALUE_NULL:
                jGenerator.writeRawValue(jParser.getText());
                break;
            case VALUE_NUMBER_FLOAT:
                // Если запись числа с trailingZeros не совпадает с записью без trailingZeros,
                // то выводим запись без trailingZeros (как это делает mysql до версии 8.04)
                // В противном случае выводим исходный токен AS IS
                // (стараемся без необходимости не менять исходный формат записи)
                BigDecimal decimalValue = jParser.getDecimalValue();
                if (!decimalValue.stripTrailingZeros().toPlainString().equals(decimalValue.toPlainString())) {
                    // При таком преобразовании всё ещё возможно расхождение с mysql при сравнении
                    // строковых представлений, но здесь всё упирается в конкретный алгоритм, которым
                    // mysql получает это строковое представление. А именно, начиная с каких чисел
                    // mysql начинает их выводить в Е-нотации. Пока здесь не меняем, так как ожидается, что
                    // таких расхождений быть почти не должно, ну и фактически данные не различаются, различаются
                    // лишь их строковые представления. Документация по этой теме
                    // https://dev.mysql.com/doc/internals/en/floating-point-types.html
                    jGenerator.writeRawValue(decimalValue.stripTrailingZeros().toPlainString());
                } else {
                    jGenerator.writeRawValue(jParser.getText());
                }
                break;
            case START_ARRAY:
                processArray(jParser, jGenerator);
                break;
            case START_OBJECT:
                processObject(jParser, jGenerator);
                break;
            default:
                throw new Mysql57CompatibleJsonFormatException("Unexpected token: " + token);
        }
    }

    private static void processObject(JsonParser jParser, JsonGenerator jGenerator) throws IOException {
        Preconditions.checkState(jParser.currentToken() == JsonToken.START_OBJECT);
        jGenerator.writeStartObject();
        while (jParser.nextToken() != JsonToken.END_OBJECT) {
            if (jParser.currentToken() == null) {
                throw new Mysql57CompatibleJsonFormatException("Unexpected EOF");
            }
            if (jParser.currentToken() == JsonToken.FIELD_NAME) {
                jGenerator.writeFieldName(jParser.currentName());
                jParser.nextToken();
                processValue(jParser, jGenerator);
            }
        }
        jGenerator.writeEndObject();
    }

    private static void processArray(JsonParser jParser, JsonGenerator jGenerator) throws IOException {
        Preconditions.checkState(jParser.currentToken() == JsonToken.START_ARRAY);
        jGenerator.writeStartArray();
        while (jParser.nextToken() != JsonToken.END_ARRAY) {
            if (jParser.currentToken() == null) {
                throw new Mysql57CompatibleJsonFormatException("Unexpected EOF");
            }
            processValue(jParser, jGenerator);
        }
        jGenerator.writeEndArray();
    }
}
