package ru.yandex.direct.mysql.slowlog.parser;

import java.time.Instant;
import java.time.format.DateTimeFormatter;
import java.util.Map;

public class SlowLogRawRecordParser {
    private static final String USER_NAME_FIELD_NAME = "User_name";
    private static final String USER_SECOND_NAME_FIELD_NAME = "User_second_name";
    private static final String USER_HOST_FIELD_NAME = "User_Host";
    private static final String USER_IP_FIELD_NAME = "User_Ip";

    private static final String YES = "Yes";
    public static final String EMPTY = "";
    private static final String ZERO = "0";

    private static final String TIME = "Time";
    private static final String USER_AND_HOST_FIELD_NAME = "User@Host";
    private static final String CONNECTION_ID = "Id";
    private static final String SCHEMA = "Schema";
    private static final String LAST_ERRNO = "Last_errno";
    private static final String KILLED = "Killed";
    private static final String QUERY_TIME = "Query_time";
    private static final String LOCK_TIME = "Lock_time";
    private static final String ROWS_SENT = "Rows_sent";
    private static final String ROWS_EXAMINED = "Rows_examined";
    private static final String ROWS_AFFECTED = "Rows_affected";
    private static final String BYTES_SENT = "Bytes_sent";
    private static final String TMP_TABLES_COUNT = "Tmp_tables";
    private static final String TMP_DISK_TABLES_COUNT = "Tmp_disk_tables";
    private static final String TMP_TABLE_SIZES = "Tmp_table_sizes";
    private static final String INNODB_TRX_ID = "InnoDB_trx_id";
    private static final String QC_HIT = "QC_Hit";
    private static final String FULL_SCAN = "Full_scan";
    private static final String FULL_JOIN = "Full_join";
    private static final String TMP_TABLE = "Tmp_table";
    private static final String TMP_TABLE_ON_DISK = "Tmp_table_on_disk";
    private static final String FILE_SORT = "Filesort";
    private static final String FILE_SORT_ON_DISK = "Filesort_on_disk";
    private static final String MERGE_PASSES = "Merge_passes";
    private static final String INNODB_IO_R_OPS = "InnoDB_IO_r_ops";
    private static final String INNODB_IO_R_BYTES = "InnoDB_IO_r_bytes";
    private static final String INNODB_IO_R_WAIT = "InnoDB_IO_r_wait";
    private static final String INNODB_REC_LOCK_WAIT = "InnoDB_rec_lock_wait";
    private static final String INNODB_QUEUE_WAIT = "InnoDB_queue_wait";
    private static final String INNODB_PAGES_DISTINCT = "InnoDB_pages_distinct";

    /**
     * Парсит запись mySQL slow query лога версии 5.7 с расширениями percona. Такая запись выглядит примерно так:
     * <pre>
     * # Time: 2021-10-07T00:06:25.879595+03:00
     * # User@Host: ppc[ppc] @ direct-perl-scripts-man-yp-1.man.yp-c.yandex.net [2a02:6b8:c0b:1996:0:648:fa3:0]  Id:
     * 37856411
     * # Schema: ppc  Last_errno: 1260  Killed: 0
     * # Query_time: 10.561406  Lock_time: 0.000286  Rows_sent: 203733  Rows_examined: 4375576  Rows_affected: 0
     * # Bytes_sent: 13023285  Tmp_tables: 3  Tmp_disk_tables: 1  Tmp_table_sizes: 16384
     * # InnoDB_trx_id: 0
     * # QC_Hit: No  Full_scan: Yes  Full_join: No  Tmp_table: Yes  Tmp_table_on_disk: Yes
     * # Filesort: No  Filesort_on_disk: No  Merge_passes: 0
     * #   InnoDB_IO_r_ops: 0  InnoDB_IO_r_bytes: 0  InnoDB_IO _r_wait: 0.000000
     * #   InnoDB_rec_lock_wait: 0.000000  InnoDB_queue_wait: 0.000000
     * #   InnoDB_pages_distinct: 8191
     * SET timestamp=1633554385;
     * ...QUERY...
     * </pre>
     * @param rawText исходный текст записи slow query лога
     * @param paramsMap карта полей записи slow query лога. Передается в параметре, чтобы каждый раз не создавать новую
     *                  (для повышения производительности и уменьшения нагрузки на gc)
     * @return распарсенная запись slow query лога
     */
    public static ParsedSlowLogRawRecord parseRawRecordText(String rawText, Map<String, String> paramsMap) {
        String prefix = "timestamp=";
        String suffix = ";\n";
        int index = rawText.indexOf(prefix);
        String queryText;
        if (index >= 0) {
            int index2 = rawText.indexOf(suffix, index + prefix.length());
            queryText = rawText.substring(index2 + suffix.length());
            // сдвигаем индекс назад, пропуская "\nSET "
            index -= 5;
        } else {
            index = rawText.length();
            queryText = "#Empty";
        }
        parseFields(rawText, index, paramsMap);
        parseUserAndHost(paramsMap);

        Instant timestamp = DateTimeFormatter.ISO_OFFSET_DATE_TIME.parse(paramsMap.get(TIME), Instant::from);

        return ParsedSlowLogRawRecord.builder()
                .withTimestamp(timestamp)
                .withBytesSentCount(Long.parseLong(paramsMap.getOrDefault(BYTES_SENT, ZERO)))
                .withConnectionId(Long.parseLong(paramsMap.getOrDefault(CONNECTION_ID, ZERO).trim()))
                .withHasFileSort(YES.equals(paramsMap.getOrDefault(FILE_SORT, EMPTY)))
                .withHasFileSortOnDisk(YES.equals(paramsMap.getOrDefault(FILE_SORT_ON_DISK, EMPTY)))
                .withHasFullJoin(YES.equals(paramsMap.getOrDefault(FULL_JOIN, EMPTY)))
                .withHasFullScan(YES.equals(paramsMap.getOrDefault(FULL_SCAN, EMPTY)))
                .withHasQcHit(YES.equals(paramsMap.getOrDefault(QC_HIT, EMPTY)))
                .withHasTempTables(YES.equals(paramsMap.getOrDefault(TMP_TABLE, EMPTY)))
                .withHasTempTablesOnDisk(YES.equals(paramsMap.getOrDefault(TMP_TABLE_ON_DISK, EMPTY)))
                .withInnoDbIoReadBytesCount(Long.parseLong(paramsMap.getOrDefault(INNODB_IO_R_BYTES, ZERO)))
                .withInnoDbIoReadOperationsCount(Integer.parseInt(paramsMap.getOrDefault(INNODB_IO_R_OPS, ZERO)))
                .withInnoDbPagesCountDistinct(Integer.parseInt(paramsMap.getOrDefault(INNODB_PAGES_DISTINCT, ZERO)))
                .withInnoDbReadWaitInSeconds(Double.parseDouble(paramsMap.getOrDefault(INNODB_IO_R_WAIT, ZERO)))
                .withInnoDbRecordsLockWaitInSeconds(
                        Double.parseDouble(paramsMap.getOrDefault(INNODB_REC_LOCK_WAIT, ZERO)))
                .withInnoQbQueueWaitInSeconds(Double.parseDouble(paramsMap.getOrDefault(INNODB_QUEUE_WAIT, ZERO)))
                .withKilledCode(Integer.parseInt(paramsMap.getOrDefault(KILLED, ZERO)))
                .withLastErrorNumber(Integer.parseInt(paramsMap.getOrDefault(LAST_ERRNO, ZERO)))
                .withLockTimeInSeconds(Double.parseDouble(paramsMap.getOrDefault(LOCK_TIME, ZERO)))
                .withMergePassesCount(Integer.parseInt(paramsMap.getOrDefault(MERGE_PASSES, ZERO)))
                .withQueryText(queryText)
                .withQueryTimeInSeconds(Double.parseDouble(paramsMap.getOrDefault(QUERY_TIME, ZERO)))
                .withRowsAffectedCount(Long.parseLong(paramsMap.getOrDefault(ROWS_AFFECTED, ZERO)))
                .withRowsExaminedCount(Long.parseLong(paramsMap.getOrDefault(ROWS_EXAMINED, ZERO)))
                .withRowsSentCount(Long.parseLong(paramsMap.getOrDefault(ROWS_SENT, ZERO)))
                .withSchema(paramsMap.getOrDefault(SCHEMA, EMPTY))
                .withTempTablesCount(Integer.parseInt(paramsMap.getOrDefault(TMP_TABLES_COUNT, ZERO)))
                .withTempTablesOnDiskCount(Integer.parseInt(paramsMap.getOrDefault(TMP_DISK_TABLES_COUNT, ZERO)))
                .withTempTablesSizesInBytes(Long.parseLong(paramsMap.getOrDefault(TMP_TABLE_SIZES, ZERO)))
                .withTransactionId(Long.parseLong(paramsMap.getOrDefault(INNODB_TRX_ID, ZERO), 16))
                .withUserHost(paramsMap.getOrDefault(USER_HOST_FIELD_NAME, EMPTY))
                .withUserIp(paramsMap.getOrDefault(USER_IP_FIELD_NAME, EMPTY))
                .withUserName(paramsMap.getOrDefault(USER_NAME_FIELD_NAME, EMPTY))
                .withUserSecondName(paramsMap.getOrDefault(USER_SECOND_NAME_FIELD_NAME, EMPTY))
                .build();
    }

    /**
     * Парсит поле User@Host из карты полей записи slow query лога, разбивает его на несколько других полей
     * и сохраняет их в ту же самую карту
     * @param paramsMap заполненная карта полей записи slow query лога
     */
    private static void parseUserAndHost(Map<String, String> paramsMap) {
        // User@Host: first_name[second_name] @ host [ip]
        String userAndHost = paramsMap.get(USER_AND_HOST_FIELD_NAME);
        if (userAndHost == null) {
            return;
        }
        int index1 = userAndHost.indexOf('[');
        paramsMap.put(USER_NAME_FIELD_NAME, userAndHost.substring(0, index1));
        index1++;
        int index2 = userAndHost.indexOf(']', index1);
        paramsMap.put(USER_SECOND_NAME_FIELD_NAME, userAndHost.substring(index1, index2));
        index1 = index2 + 4;
        index2 = userAndHost.indexOf(' ', index1);
        paramsMap.put(USER_HOST_FIELD_NAME, userAndHost.substring(index1, index2));
        paramsMap.put(USER_IP_FIELD_NAME, userAndHost.substring(index2 + 2, userAndHost.length() - 1));
    }

    /**
     * Парсит все поля записи slow query лога и сохраняет их в карту полей paramsMap
     * @param text исходный текст записи
     * @param pos конец описательной части записи, перед первым SET timestamp
     * @param paramsMap карта полей записи slow query лога, куда будут сохраняться найденные поля и их значения
     */
    private static void parseFields(String text, int pos, Map<String, String> paramsMap) {
        paramsMap.clear();
        boolean valueStage = true;
        int valueEndPos = pos;
        if (pos == text.length()) {
            pos--;
        }
        int valueBeginPos = pos;
        char lastChar = '\n';
        while (pos >= 0) {
            char c = text.charAt(pos);
            if (valueStage) {
                if (c == ':' && lastChar == ' ') {
                    valueStage = false;
                    valueBeginPos = pos;
                } else if (c == '#' || c == '\n') {
                    valueEndPos = pos;
                }
            } else {
                if (c == ' ') {
                    valueStage = true;
                    String key = text.substring(pos + 1, valueBeginPos);
                    String value = text.substring(valueBeginPos + 2, valueEndPos);
                    paramsMap.put(key, value);
                    pos--;
                    valueEndPos = pos;
                }
            }
            pos--;
            lastChar = c;
        }
    }
}
