package ru.yandex.direct.mysql;

import java.io.IOException;
import java.sql.SQLException;
import java.time.Duration;

import ru.yandex.direct.utils.Interrupts;
import ru.yandex.direct.utils.MonotonicTime;
import ru.yandex.direct.utils.NanoTimeClock;

public class AsyncStreamer implements AutoCloseable {
    private MySQLBinlogDataStreamer streamer;
    private MySQLBinlogConsumer consumer;
    private Thread background;
    private volatile Exception backgroundException;
    private ExitMonitor exitMonitor;

    public AsyncStreamer(MySQLBinlogDataStreamer streamer, MySQLBinlogConsumer consumer, MySQLInstance mysql) {
        this(streamer, consumer, mysql, null);
    }

    public AsyncStreamer(MySQLBinlogDataStreamer streamer, MySQLBinlogConsumer consumer, MySQLInstance mysql,
                         ExitMonitor exitMonitor) {
        this.streamer = streamer;
        this.consumer = consumer;
        this.backgroundException = null;
        this.exitMonitor = exitMonitor;
        this.background = new Thread(() -> {
            try {
                streamer.run(this.consumer, mysql);
            } catch (IOException | SQLException | RuntimeException exc) {
                backgroundException = exc;
                if (this.exitMonitor != null) {
                    this.exitMonitor.failed();
                }
            } finally {
                if (this.exitMonitor != null) {
                    this.exitMonitor.done();
                }
            }
        });
        this.background.setName("Streamer-" + this.background.getName());
        this.background.start();
    }

    public MySQLBinlogConsumer getConsumer() {
        return consumer;
    }

    public void stop() {
        streamer.stop();
    }

    @Override
    public void close() {
        stop();
        Interrupts.criticalTimeoutWait(
                Duration.ofSeconds(60),
                timeout -> {
                    Interrupts.waitMillisNanos(background::join).await(timeout); // IS-NOT-COMPLETABLE-FUTURE-JOIN
                    return !background.isAlive();
                }
        );
        if (backgroundException != null) {
            throw new RuntimeException("MySQL streamer exception in " + streamer, backgroundException);
        }
    }

    @Override
    public String toString() {
        return "AsyncStreamer{" +
                "streamer=" + streamer +
                '}';
    }

    public static class ExitMonitor {
        private int workersCount;

        public ExitMonitor(int workersCount) {
            this.workersCount = workersCount;
        }

        synchronized void done() {
            workersCount -= 1;
            if (workersCount <= 0) {
                notifyAll();
            }
        }

        synchronized void failed() {
            workersCount = 0;
            notifyAll();
        }

        public synchronized boolean awaitExit(Duration timeout) throws InterruptedException {
            MonotonicTime now = NanoTimeClock.now();
            MonotonicTime deadline = now.plus(timeout);
            while (now.isBefore(deadline) && workersCount > 0) {
                Interrupts.waitMillisNanos(this::wait).await(deadline.minus(now));
                now = NanoTimeClock.now();
            }
            return workersCount <= 0;
        }
    }
}
