package ru.yandex.direct.mysql;

import java.io.IOException;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

import com.github.shyiko.mysql.binlog.GtidSet;
import com.github.shyiko.mysql.binlog.event.DeleteRowsEventData;
import com.github.shyiko.mysql.binlog.event.Event;
import com.github.shyiko.mysql.binlog.event.EventHeaderV4;
import com.github.shyiko.mysql.binlog.event.EventType;
import com.github.shyiko.mysql.binlog.event.GtidEventData;
import com.github.shyiko.mysql.binlog.event.QueryEventData;
import com.github.shyiko.mysql.binlog.event.RowsQueryEventData;
import com.github.shyiko.mysql.binlog.event.TableMapEventData;
import com.github.shyiko.mysql.binlog.event.UpdateRowsEventData;
import com.github.shyiko.mysql.binlog.event.WriteRowsEventData;
import com.github.shyiko.mysql.binlog.event.XidEventData;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.mysql.schema.DatabaseSchema;
import ru.yandex.direct.mysql.schema.ServerSchema;
import ru.yandex.direct.mysql.schema.TableSchema;

import static ru.yandex.direct.mysql.MySQLUtils.LOG_EVENT_SUPPRESS_USE_F;

/**
 * Превращает сырой поток событий в поток наших событий
 */
public class BinlogEventConverter {
    private static final Logger logger = LoggerFactory.getLogger(BinlogEventConverter.class);

    private static final String[] DML_PREFIXES = new String[]{
            "CALL",
            "DELETE",
            "DO",
            "INSERT",
            "REPLACE",
            "SELECT",
            "UPDATE"
    };
    // префиксы DDL-запросов, которые не меняют схему (или меняют, но у системных таблиц)
    private static final String[] SKIP_DDL_PREFIXES = new String[]{
            "ALTER USER ",
            "FLUSH ",
            "GRANT ",
            "REVOKE ",
            "CREATE USER ",
            "DROP USER ",
            "TRUNCATE "
    };

    /**
     * Интервал обязательных запросов для поддержания соединения в рабочем состоянии
     */
    private static final long SCHEMA_CONNECTION_KEEP_ALIVE_INTERVAL = TimeUnit.MINUTES.toNanos(5);

    private ServerSchema processedServerSchema;
    private GtidSet processedGtidSet;
    private Connection schemaConnection = null;
    private long schemaConnectionNextPing = -1L;

    private BinlogRawEventSource rawEventSource = null;

    private boolean eventSourceChanged = false;
    private Map<Long, TableMapEventData> tableMaps = new HashMap<>();

    private Map<Long, TableSchema> tableSchemas = new HashMap<>();
    private boolean inTransaction = false;
    private String currentGtid = null;
    private String lastTransactionFlushQuery = null;
    private String lastTransactionCommitQuery = null;

    public BinlogEventConverter(MySQLBinlogState state) {
        this.processedServerSchema = state.getServerSchema();
        this.processedGtidSet = new GtidSet(state.getGtidSet());
    }

    public MySQLBinlogState getState() {
        return new MySQLBinlogState(processedServerSchema, processedGtidSet.toString());
    }

    private void maybePingSchemaConnection() throws SQLException {
        if (schemaConnectionNextPing == -1L || schemaConnectionNextPing <= System.nanoTime()) {
            try (PreparedStatement stmt = schemaConnection.prepareStatement("SELECT 1")) {
                try (ResultSet rs = stmt.executeQuery()) {
                    rs.next();
                }
            }
            markSchemaConnectionWorking();
        }
    }

    private void markSchemaConnectionWorking() {
        schemaConnectionNextPing = System.nanoTime() + SCHEMA_CONNECTION_KEEP_ALIVE_INTERVAL;
    }

    public void attachSchemaConnection(Connection schemaConnection) throws SQLException {
        this.schemaConnection = schemaConnection;
        // Выставляем наиболее совместимый с неожиданными ALTER'ами режим работы
        MySQLUtils.loosenRestrictions(schemaConnection);
        processedServerSchema.restore(schemaConnection);
        markSchemaConnectionWorking();
    }

    public BinlogRawEventSource getRawEventSource() {
        return rawEventSource;
    }

    public void attachRawEventSource(BinlogRawEventSource rawEventSource) {
        this.rawEventSource = rawEventSource;
        eventSourceChanged = true;
    }

    public boolean isRawEventSourceSet() {
        return rawEventSource != null;
    }

    private static TableSchema findTableSchema(ServerSchema schema, String databaseName, String tableName) {
        for (DatabaseSchema database : schema.getDatabases()) {
            if (database.getName().equalsIgnoreCase(databaseName)) {
                for (TableSchema table : database.getTables()) {
                    if (table.getName().equalsIgnoreCase(tableName)) {
                        return table;
                    }
                }
            }
        }
        return null;
    }

    private TableSchema findTableSchema(TableMapEventData data) {
        return findTableSchema(processedServerSchema, data.getDatabase(), data.getTable());
    }

    private TableMapEventData requireTableMap(long tableId) {
        TableMapEventData tableMap = tableMaps.get(tableId);
        if (tableMap == null) {
            throw new MySQLBinlogException("There is no table map for table with id " + tableId);
        }
        return tableMap;
    }

    private void flushTransaction(String query) {
        if (inTransaction) {
            throw new MySQLBinlogException(
                    "Found an attempt to commit implicitly while in transaction: " + query);
        }
        if (currentGtid != null) {
            processedGtidSet.add(currentGtid);
            currentGtid = null;
        }
        tableMaps.clear();
        tableSchemas.clear();
        lastTransactionFlushQuery = query;
    }

    private void beginTransaction() {
        if (inTransaction) {
            throw new IllegalStateException("Found an attempt to begin transaction while in transaction");
        }
        if (currentGtid == null) {
            throw new IllegalStateException("Found an attempt to begin transaction without gtid");
        }
        inTransaction = true;
    }

    private void endTransaction(String query) {
        if (!inTransaction) {
            throw new IllegalStateException(
                    "Found an attempt to end transaction while not in transaction: " + query + " (last end was: "
                            + lastTransactionCommitQuery + ", last flush was: " + lastTransactionFlushQuery + ")");
        }
        inTransaction = false;
        flushTransaction(query);
        lastTransactionCommitQuery = query;
    }

    private static boolean looksLikeDML(String query) {
        query = MySQLUtils.extractFirstWordFromSQL(query);
        for (String prefix : DML_PREFIXES) {
            if (prefix.equalsIgnoreCase(query)) {
                return true;
            }
        }
        return false;
    }

    public static boolean looksLikeDDL(Event rawEvent) {
        if (!EventType.QUERY.equals(rawEvent.getHeader().getEventType())) {
            return false;
        }

        String query = ((QueryEventData) rawEvent.getData()).getSql();

        if ("BEGIN".equalsIgnoreCase(query)) {
            return false;
        }
        if ("COMMIT".equalsIgnoreCase(query)) {
            return false;
        }
        if ("ROLLBACK".equalsIgnoreCase(query)) {
            return false;
        }
        if (query.startsWith("SAVEPOINT ")) {
            return false;
        }
        if (MySQLUtils.looksLikeThreadSpecific(rawEvent)) {
            return false;
        }
        if (looksLikeDML(query)) {
            return false;
        }
        return true;
    }

    public static boolean shouldApplySchemaDDL(String sql) {
        for (String prefix : SKIP_DDL_PREFIXES) {
            if (sql.toUpperCase().startsWith(prefix)) {
                return false;
            }
        }
        // Все остальные DDL применяем на базе со схемой
        return true;
    }

    private Optional<BinlogEvent> handleGTID(Event rawEvent, GtidEventData data) {
        if (inTransaction) {
            throw new IllegalStateException("Found GTID switch inside a transaction");
        }
        currentGtid = data.getGtid();
        return Optional.empty();
    }

    private Optional<BinlogEvent> handleTableMap(Event rawEvent, TableMapEventData data) {
        tableMaps.put(data.getTableId(), data);
        TableSchema table = findTableSchema(data);
        if (table != null) {
            tableSchemas.put(data.getTableId(), table);
        }
        return Optional.empty();
    }

    private Optional<BinlogEvent> handleRowsQuery(Event rawEvent, RowsQueryEventData data) {
        if (!inTransaction) {
            throw new IllegalStateException("Found ROWS_QUERY outside of a transaction");
        }
        return Optional.of(new BinlogEvent(rawEvent.getHeader().getTimestamp(), BinlogEventType.ROWS_QUERY,
                new BinlogEventData.RowsQuery(currentGtid, data)));
    }

    private Optional<BinlogEvent> handleQuery(Event rawEvent, QueryEventData data)
            throws SQLException, BinlogDDLException {
        String sql = data.getSql();
        try {
            return handleQueryGuarded(rawEvent, data, sql);
        } catch (SQLException | BinlogDDLException | RuntimeException exc) {
            logger.warn("Got error while handling query:\n{}", sql);
            throw exc;
        }
    }

    private Optional<BinlogEvent> handleQueryGuarded(Event rawEvent, QueryEventData data, String sql)
            throws SQLException, BinlogDDLException {
        if ("BEGIN".equalsIgnoreCase(sql)) {
            beginTransaction();
            return Optional.of(new BinlogEvent(rawEvent.getHeader().getTimestamp(), BinlogEventType.BEGIN,
                    new BinlogEventData.Begin(currentGtid)));
        }
        if ("COMMIT".equalsIgnoreCase(sql)) {
            String gtid = currentGtid;
            endTransaction(sql);
            return Optional.of(new BinlogEvent(rawEvent.getHeader().getTimestamp(), BinlogEventType.COMMIT,
                    new BinlogEventData.Commit(gtid)));
        }
        if ("ROLLBACK".equalsIgnoreCase(sql)) {
            String gtid = currentGtid;
            endTransaction(sql);
            return Optional.of(new BinlogEvent(rawEvent.getHeader().getTimestamp(), BinlogEventType.ROLLBACK,
                    new BinlogEventData.Rollback(gtid)));
        }
        if (sql.startsWith("SAVEPOINT ")) {
            // We don't care about savepoints
            return Optional.empty();
        }
        if (MySQLUtils.looksLikeThreadSpecific(rawEvent)) {
            // Игнорируем все thread-specific запросы (временные таблицы и т.д.)
            if (!looksLikeDML(sql)) {
                flushTransaction("/* THREAD SPECIFIC */ " + sql);
            }
            return Optional.empty();
        }
        if (looksLikeDML(sql)) {
            if (!inTransaction) {
                throw new IllegalStateException("Found DML outside of a transaction");
            }
            return Optional.of(new BinlogEvent(rawEvent.getHeader().getTimestamp(), BinlogEventType.DML,
                    new BinlogEventData.DML(currentGtid, data)));
        }
        ServerSchema before = processedServerSchema;
        ServerSchema after;
        if (shouldApplySchemaDDL(sql)) {
            // Обрабатываем DDL, применяя его на нашей базе
            EventHeaderV4 header = rawEvent.getHeader();
            if ((header.getFlags() & LOG_EVENT_SUPPRESS_USE_F) == 0 && !StringUtils.isEmpty(data.getDatabase())) {
                // Нам нужно переключить базу перед выполнением запроса
                schemaConnection.setCatalog(data.getDatabase());
            }
            try {
                try (PreparedStatement stmt = schemaConnection.prepareStatement(sql)) {
                    stmt.executeUpdate();
                }
            } catch (SQLException e) {
                throw new BinlogDDLException(sql, e);
            }
            after = ServerSchema.dump(schemaConnection);
            markSchemaConnectionWorking();
        } else {
            after = before;
        }
        String gtid = currentGtid;
        flushTransaction(sql);
        processedServerSchema = after;
        return Optional.of(new BinlogEvent(rawEvent.getHeader().getTimestamp(), BinlogEventType.DDL,
                new BinlogEventData.DDL(gtid, data, before, after)));
    }

    private Optional<BinlogEvent> handleXID(Event rawEvent, XidEventData data) {
        String gtid = currentGtid;
        endTransaction("XID COMMIT");
        return Optional.of(new BinlogEvent(rawEvent.getHeader().getTimestamp(), BinlogEventType.COMMIT,
                new BinlogEventData.Commit(gtid)));
    }

    private Optional<BinlogEvent> handleWriteRows(Event rawEvent, WriteRowsEventData data) {
        if (!inTransaction) {
            throw new IllegalStateException("Found WRITE_ROWS outside of a transaction");
        }
        TableMapEventData tableMap = requireTableMap(data.getTableId());
        TableSchema tableSchema = tableSchemas.get(data.getTableId());
        return Optional.of(new BinlogEvent(rawEvent.getHeader().getTimestamp(), BinlogEventType.INSERT,
                new BinlogEventData.Insert(currentGtid, data, tableMap, tableSchema)));
    }

    private Optional<BinlogEvent> handleUpdateRows(Event rawEvent, UpdateRowsEventData data) {
        if (!inTransaction) {
            throw new IllegalStateException("Found UPDATE_ROWS outside of a transaction");
        }
        TableMapEventData tableMap = requireTableMap(data.getTableId());
        TableSchema tableSchema = tableSchemas.get(data.getTableId());
        return Optional.of(new BinlogEvent(rawEvent.getHeader().getTimestamp(), BinlogEventType.UPDATE,
                new BinlogEventData.Update(currentGtid, data, tableMap, tableSchema)));
    }

    private Optional<BinlogEvent> handleDeleteRows(Event rawEvent, DeleteRowsEventData data) {
        if (!inTransaction) {
            throw new IllegalStateException("Found DELETE_ROWS outside of a transaction");
        }
        TableMapEventData tableMap = requireTableMap(data.getTableId());
        TableSchema tableSchema = tableSchemas.get(data.getTableId());
        return Optional.of(new BinlogEvent(rawEvent.getHeader().getTimestamp(), BinlogEventType.DELETE,
                new BinlogEventData.Delete(currentGtid, data, tableMap, tableSchema)));
    }

    private Optional<BinlogEvent> handleEvent(Event rawEvent) throws SQLException, BinlogDDLException {
        switch (rawEvent.getHeader().getEventType()) {
            case FORMAT_DESCRIPTION:
            case PREVIOUS_GTIDS:
            case HEARTBEAT:
            case ROTATE:
                // Нам пока не интересны данные события
                return Optional.empty();
            case TABLE_MAP:
                return handleTableMap(rawEvent, rawEvent.getData());
            case GTID:
                return handleGTID(rawEvent, rawEvent.getData());
            case ROWS_QUERY:
                return handleRowsQuery(rawEvent, rawEvent.getData());
            case QUERY:
                return handleQuery(rawEvent, rawEvent.getData());
            case XID:
                return handleXID(rawEvent, rawEvent.getData());
            case WRITE_ROWS:
            case EXT_WRITE_ROWS:
                return handleWriteRows(rawEvent, rawEvent.getData());
            case UPDATE_ROWS:
            case EXT_UPDATE_ROWS:
                return handleUpdateRows(rawEvent, rawEvent.getData());
            case DELETE_ROWS:
            case EXT_DELETE_ROWS:
                return handleDeleteRows(rawEvent, rawEvent.getData());
            default:
                return Optional.empty();
        }
    }

    public BinlogEvent readEvent(long timeout, TimeUnit unit)
            throws InterruptedException, TimeoutException, IOException, SQLException, BinlogDDLException {
        maybePingSchemaConnection();
        if (eventSourceChanged) {
            eventSourceChanged = false;
            tableMaps.clear();
            tableSchemas.clear();
            if (inTransaction) {
                BinlogEvent rollback = new BinlogEvent(
                        0,
                        BinlogEventType.ROLLBACK,
                        new BinlogEventData.Rollback(currentGtid));
                inTransaction = false;
                currentGtid = null;
                return rollback;
            }
        }
        Event rawEvent;
        while ((rawEvent = rawEventSource.readEvent(timeout, unit)) != null) {
            Optional<BinlogEvent> event = handleEvent(rawEvent);
            if (event.isPresent()) {
                return event.get();
            }
            maybePingSchemaConnection();
        }
        return null;
    }
}
