package ru.yandex.direct.mysql;

import java.io.Serializable;
import java.util.Collections;
import java.util.List;

import com.github.shyiko.mysql.binlog.event.DeleteRowsEventData;
import com.github.shyiko.mysql.binlog.event.QueryEventData;
import com.github.shyiko.mysql.binlog.event.RowsQueryEventData;
import com.github.shyiko.mysql.binlog.event.TableMapEventData;
import com.github.shyiko.mysql.binlog.event.UpdateRowsEventData;
import com.github.shyiko.mysql.binlog.event.WriteRowsEventData;

import ru.yandex.direct.mysql.schema.ColumnSchema;
import ru.yandex.direct.mysql.schema.ServerSchema;
import ru.yandex.direct.mysql.schema.TableSchema;

public interface BinlogEventData {
    String getGtid();

    class DDL implements BinlogEventData {
        private final String gtid;
        private final QueryEventData data;
        private final ServerSchema before;
        private final ServerSchema after;

        public DDL(String gtid, QueryEventData data, ServerSchema before, ServerSchema after) {
            this.gtid = gtid;
            this.data = data;
            this.before = before;
            this.after = after;
        }

        public String getGtid() {
            return gtid;
        }

        public QueryEventData getData() {
            return data;
        }

        public ServerSchema getBefore() {
            return before;
        }

        public ServerSchema getAfter() {
            return after;
        }

        @Override
        public String toString() {
            return data.getSql();
        }
    }

    class DML implements BinlogEventData {
        private final String gtid;
        private final QueryEventData data;

        public DML(String gtid, QueryEventData data) {
            this.gtid = gtid;
            this.data = data;
        }

        public String getGtid() {
            return gtid;
        }

        public QueryEventData getData() {
            return data;
        }

        @Override
        public String toString() {
            return data.getSql();
        }
    }

    class Begin implements BinlogEventData {
        private final String gtid;

        public Begin(String gtid) {
            this.gtid = gtid;
        }

        public String getGtid() {
            return gtid;
        }

        @Override
        public String toString() {
            return "BEGIN";
        }
    }

    class RowsQuery implements BinlogEventData {
        private final String gtid;
        private final RowsQueryEventData data;

        public RowsQuery(String gtid, RowsQueryEventData data) {
            this.gtid = gtid;
            this.data = data;
        }

        public String getGtid() {
            return gtid;
        }

        public RowsQueryEventData getData() {
            return data;
        }

        @Override
        public String toString() {
            return data.getQuery();
        }
    }

    interface BinlogRowsEventData extends BinlogEventData {
        TableMapEventData getTableMap();
    }

    class Insert implements BinlogRowsEventData {
        private static final List<ColumnSchema> EMPTY_COLUMNS = Collections.emptyList();
        private static final List<Serializable[]> EMPTY_ROWS = Collections.singletonList(new Serializable[0]);

        private final String gtid;
        private final WriteRowsEventData data;
        private final TableMapEventData tableMap;
        private final TableSchema tableSchema;

        public Insert(String gtid, WriteRowsEventData data, TableMapEventData tableMap, TableSchema tableSchema) {
            this.gtid = gtid;
            this.data = data;
            this.tableMap = tableMap;
            this.tableSchema = tableSchema;
        }

        public String getGtid() {
            return gtid;
        }

        public WriteRowsEventData getData() {
            return data;
        }

        @Override
        public TableMapEventData getTableMap() {
            return tableMap;
        }

        public TableSchema getTableSchema() {
            return tableSchema;
        }

        public MySQLSimpleRows getRows() {
            List<ColumnSchema> columns =
                    BinlogEventDataUtil.columnsFromTableMap(tableMap, tableSchema, data.getIncludedColumns());
            List<ColumnSchema> defaultColumns =
                    BinlogEventDataUtil.columnsFromTableMap(tableMap, tableSchema, data.getIncludedColumns(), false);
            List<Serializable[]> rows = data.getRows();
            if (rows.isEmpty()) {
                // MySQL использует пустой INSERT для вставки строк полностью состоящих из колонок по умолчанию
                return new MySQLSimpleRows(EMPTY_COLUMNS, EMPTY_ROWS, defaultColumns);
            }
            return new MySQLSimpleRows(columns, rows, defaultColumns);
        }
    }

    class Update implements BinlogRowsEventData {
        private final String gtid;
        private final UpdateRowsEventData data;
        private final TableMapEventData tableMap;
        private final TableSchema tableSchema;

        public Update(String gtid, UpdateRowsEventData data, TableMapEventData tableMap, TableSchema tableSchema) {
            this.gtid = gtid;
            this.data = data;
            this.tableMap = tableMap;
            this.tableSchema = tableSchema;
        }

        public String getGtid() {
            return gtid;
        }

        public UpdateRowsEventData getData() {
            return data;
        }

        @Override
        public TableMapEventData getTableMap() {
            return tableMap;
        }

        public TableSchema getTableSchema() {
            return tableSchema;
        }

        public MySQLUpdateRows getRows() {
            List<ColumnSchema> beforeColumns = BinlogEventDataUtil
                    .columnsFromTableMap(tableMap, tableSchema, data.getIncludedColumnsBeforeUpdate());
            List<ColumnSchema> afterColumns =
                    BinlogEventDataUtil.columnsFromTableMap(tableMap, tableSchema, data.getIncludedColumns());
            return new MySQLUpdateRows(beforeColumns, afterColumns, data.getRows());
        }
    }

    class Delete implements BinlogRowsEventData {
        private final String gtid;
        private final DeleteRowsEventData data;
        private final TableMapEventData tableMap;
        private final TableSchema tableSchema;

        public Delete(String gtid, DeleteRowsEventData data, TableMapEventData tableMap, TableSchema tableSchema) {
            this.gtid = gtid;
            this.data = data;
            this.tableMap = tableMap;
            this.tableSchema = tableSchema;
        }

        public String getGtid() {
            return gtid;
        }

        public DeleteRowsEventData getData() {
            return data;
        }

        @Override
        public TableMapEventData getTableMap() {
            return tableMap;
        }

        public TableSchema getTableSchema() {
            return tableSchema;
        }

        public MySQLSimpleRows getRows() {
            List<ColumnSchema> columns =
                    BinlogEventDataUtil.columnsFromTableMap(tableMap, tableSchema, data.getIncludedColumns());
            return new MySQLSimpleRows(columns, data.getRows());
        }
    }

    class Commit implements BinlogEventData {
        private final String gtid;

        public Commit(String gtid) {
            this.gtid = gtid;
        }

        public String getGtid() {
            return gtid;
        }

        @Override
        public String toString() {
            return "COMMIT";
        }
    }

    class Rollback implements BinlogEventData {
        private final String gtid;

        public Rollback(String gtid) {
            this.gtid = gtid;
        }

        public String getGtid() {
            return gtid;
        }

        @Override
        public String toString() {
            return "ROLLBACK";
        }
    }
}
