package ru.yandex.direct.mysql;

import java.io.IOException;
import java.util.ArrayDeque;
import java.util.Deque;
import java.util.concurrent.CompletionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

import com.github.shyiko.mysql.binlog.BinaryLogClient;
import com.github.shyiko.mysql.binlog.event.Event;
import com.github.shyiko.mysql.binlog.event.deserialization.EventDeserializer;

public class BinlogRawEventServerSource implements BinlogRawEventSource {
    private final Deque<Event> events = new ArrayDeque<>();
    private final int maxEvents;
    private final EventDeserializer eventDeserializer;
    private final BinaryLogClient binaryLogClient;
    private final Lock lock = new ReentrantLock();
    private final Condition canRead = lock.newCondition();
    private final Condition canWrite = lock.newCondition();
    private final ReaderThread readerThread;
    private Throwable error = null;
    private boolean closed = false;

    public BinlogRawEventServerSource(String host, int port, String username, String password, int serverId,
                                      String gtidSet, int maxBufferedEvents) {
        maxEvents = maxBufferedEvents;
        eventDeserializer = new EventDeserializer();
        binaryLogClient = new BinaryLogClient(host, port, username, password);
        binaryLogClient.setServerId(serverId);
        binaryLogClient.setGtidSet(gtidSet);

        // Хочешь включить keep-alive? Одумайся! Тебя ждут дикие приключения в джунглях BinaryLogClient.
        // DIRECT-78730
        // DIRECT-79059
        // DIRECT-79147
        // DIRECT-79188
        // DIRECT-79453
        // DIRECT-79491
        // DIRECT-79589
        // Была даже попытка переписать этот класс совсем.
        // DIRECT-79436
        // Завершилась провалом.
        // DIRECT-80494
        // DIRECT-80532
        binaryLogClient.setKeepAlive(false);

        binaryLogClient.setEventDeserializer(eventDeserializer);
        binaryLogClient.registerLifecycleListener(new LifecycleListener());
        binaryLogClient.registerEventListener(new EventListener());
        readerThread = new ReaderThread("binlog-reader-" + host + ":" + port);
        readerThread.start();
    }

    @Override
    public Event readEvent(long timeout, TimeUnit unit) throws IOException, TimeoutException, InterruptedException {
        long deadline = System.nanoTime() + unit.toNanos(timeout);
        lock.lockInterruptibly();
        try {
            long delay = deadline - System.nanoTime();
            while (events.isEmpty() && !closed) {
                if (delay <= 0) {
                    throw new TimeoutException();
                }
                delay = canRead.awaitNanos(delay);
            }
            if (!events.isEmpty()) {
                Event event = events.removeFirst();
                canWrite.signal();
                return event;
            }
            if (error != null) {
                if (error instanceof RuntimeException) {
                    throw (RuntimeException) error;
                }
                if (error instanceof Error) {
                    throw (Error) error;
                }
                if (error instanceof IOException) {
                    throw (IOException) error;
                }
                throw new CompletionException(error);
            }
            return null;
        } finally {
            lock.unlock();
        }
    }

    @Override
    public void close() throws IOException {
        try {
            lock.lock();
            try {
                closed = true;
                events.clear();
                canRead.signalAll();
                canWrite.signalAll();
                readerThread.interrupt();
            } finally {
                lock.unlock();
            }
        } finally {
            binaryLogClient.disconnect();
        }
    }

    private boolean isClosed() {
        lock.lock();
        try {
            return closed;
        } finally {
            lock.unlock();
        }
    }

    private void closeWithError(Throwable e) {
        lock.lock();
        try {
            if (!closed) {
                error = e;
                closed = true;
                canRead.signalAll();
                canWrite.signalAll();
            }
        } finally {
            lock.unlock();
        }
    }

    private class ReaderThread extends Thread {
        public ReaderThread(String name) {
            setName(name);
            setDaemon(true);
        }

        @Override
        public void run() {
            if (isClosed()) {
                return;
            }
            Throwable error = null;
            try {
                binaryLogClient.connect();
            } catch (Throwable e) {
                error = e;
            } finally {
                closeWithError(error);
            }
        }
    }

    private class LifecycleListener implements BinaryLogClient.LifecycleListener {
        @Override
        public void onConnect(BinaryLogClient client) {
            if (isClosed()) {
                // Клиент закрыли до успешного connect'а
                // DIRECT-79059
                // Следует предотвратить возможность запуска keep-alive потока после
                // разрыва соединения.
                binaryLogClient.setKeepAlive(false);
                try {
                    binaryLogClient.disconnect();
                } catch (IOException ignored) {
                    // Нам уже всё-равно
                }
            }
        }

        @Override
        public void onCommunicationFailure(BinaryLogClient client, Exception ex) {
            closeWithError(ex);
        }

        @Override
        public void onEventDeserializationFailure(BinaryLogClient client, Exception ex) {
            closeWithError(ex);
            try {
                client.disconnect();
            } catch (IOException e) {
                // Нам уже всё-равно
            }
        }

        @Override
        public void onDisconnect(BinaryLogClient client) {
            // Вызывается при любом disconnect'е
            // Мы всё это уже ловим на выходе из connect
        }
    }

    private class EventListener implements BinaryLogClient.EventListener {
        @Override
        public void onEvent(Event event) {
            lock.lock();
            try {
                while (!closed && events.size() >= maxEvents) {
                    canWrite.awaitUninterruptibly();
                }
                if (closed) {
                    // Если нас закрыли, то игнорируем поток, пока он есть
                    return;
                }
                events.addLast(event);
                canRead.signal();
            } finally {
                lock.unlock();
            }
        }
    }
}
