package ru.yandex.direct.mysql;

import java.io.IOException;
import java.time.Duration;
import java.util.concurrent.CountDownLatch;

import com.github.shyiko.mysql.binlog.BinaryLogClient;
import com.github.shyiko.mysql.binlog.event.deserialization.EventDeserializer;

import ru.yandex.direct.utils.Interrupts;

/**
 * Data provider that uses an actual mysql client
 */
public class MySQLBinlogDataClientProvider implements MySQLBinlogDataProvider {
    private String host;
    private int port;
    private String username;
    private String password;
    private int serverId;
    private BinaryLogClient client = null;
    private boolean stopped;
    private CountDownLatch connectedLatch;

    public MySQLBinlogDataClientProvider(String host, int port, String username, String password, int serverId) {
        this.host = host;
        this.port = port;
        this.username = username;
        this.password = password;
        this.serverId = serverId;
        this.stopped = false;
        this.connectedLatch = new CountDownLatch(1);
    }

    public String getHost() {
        return host;
    }

    public int getPort() {
        return port;
    }

    public String getUsername() {
        return username;
    }

    public String getPassword() {
        return password;
    }

    @Override
    public void run(String gtidSet, BinaryLogClient.LifecycleListener lifecycleListener,
                    BinaryLogClient.EventListener eventListener)
            throws IOException {
        EventDeserializer eventDeserializer = new EventDeserializer();
        synchronized (this) {
            if (!stopped) {
                if (client != null) {
                    throw new MySQLBinlogException("Provider is already running");
                }
                client = new BinaryLogClient(host, port, username, password);
                client.setServerId(serverId);
                client.setGtidSet(gtidSet);
                client.setKeepAlive(false);
                client.setEventDeserializer(eventDeserializer);
                client.registerLifecycleListener(new BinaryLogClient.LifecycleListener() {
                    @Override
                    public void onConnect(BinaryLogClient client) {
                        connectedLatch.countDown();
                        lifecycleListener.onConnect(client);
                    }

                    @Override
                    public void onCommunicationFailure(BinaryLogClient client, Exception ex) {
                        lifecycleListener.onCommunicationFailure(client, ex);
                    }

                    @Override
                    public void onEventDeserializationFailure(BinaryLogClient client, Exception ex) {
                        lifecycleListener.onEventDeserializationFailure(client, ex);
                    }

                    @Override
                    public void onDisconnect(BinaryLogClient client) {
                        lifecycleListener.onDisconnect(client);
                    }
                });
                client.registerEventListener(eventListener);
            }
        }
        // client == null только если был вызван stop
        if (client != null) {
            try {
                client.connect();
            } finally {
                synchronized (this) {
                    client = null;
                    // client.connect() может завершится с ошибкой еще до того как случится onConnect,
                    // в таком случае нужно просигнализировать, что ждать в stop больше не нужно.
                    connectedLatch.countDown();
                }
            }
        }
    }

    @Override
    public void stop() {
        try {
            boolean localStopped;
            BinaryLogClient localClient;
            synchronized (this) {
                localStopped = stopped;
                if (!stopped) {
                    stopped = true;
                }
                localClient = client;
            }
            if (!localStopped && localClient != null) {
                // client.connect() может надолго застрять в Socket.connect(), поэтому нужно дать ему время.
                // В линухе максимальный таймаут на connect - 20 сек. поэтому ждем с небольшим запасом (30 сек).
                Interrupts.criticalTimeoutWait(Duration.ofSeconds(30), (Interrupts.WaitTimeUnit) connectedLatch::await);
                localClient.disconnect();
            }
        } catch (IOException e) {
            throw new MySQLBinlogException("Disconnect failed", e);
        }
    }

    @Override
    public String toString() {
        return "MySQLBinlogDataClientProvider{" +
                "host='" + host + '\'' +
                ", port=" + port +
                ", username='" + username + '\'' +
                ", serverId=" + serverId +
                '}';
    }
}
