package ru.yandex.direct.mysql;

import java.io.IOException;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.util.HashMap;
import java.util.Map;

import com.github.shyiko.mysql.binlog.BinaryLogClient;
import com.github.shyiko.mysql.binlog.GtidSet;
import com.github.shyiko.mysql.binlog.event.DeleteRowsEventData;
import com.github.shyiko.mysql.binlog.event.Event;
import com.github.shyiko.mysql.binlog.event.EventHeaderV4;
import com.github.shyiko.mysql.binlog.event.GtidEventData;
import com.github.shyiko.mysql.binlog.event.QueryEventData;
import com.github.shyiko.mysql.binlog.event.RowsQueryEventData;
import com.github.shyiko.mysql.binlog.event.TableMapEventData;
import com.github.shyiko.mysql.binlog.event.UpdateRowsEventData;
import com.github.shyiko.mysql.binlog.event.WriteRowsEventData;
import org.apache.commons.lang3.StringUtils;

import ru.yandex.direct.mysql.schema.DatabaseSchema;
import ru.yandex.direct.mysql.schema.ServerSchema;
import ru.yandex.direct.mysql.schema.TableSchema;

import static ru.yandex.direct.mysql.MySQLUtils.LOG_EVENT_SUPPRESS_USE_F;
import static ru.yandex.direct.mysql.MySQLUtils.looksLikeThreadSpecific;

/**
 * MySQL binlog data streamer with temporary database and schema detection
 */
public class MySQLBinlogDataStreamer {
    private static final String[] DML_PREFIXES = new String[]{
            "CALL",
            "DELETE",
            "DO",
            "INSERT",
            "REPLACE",
            "SELECT",
            "UPDATE"
    };

    private final MySQLBinlogDataProvider provider;
    private ServerSchema processedServerSchema = null;
    private GtidSet processedGtidSet = null;
    private boolean running = false;
    private Throwable failure = null;

    public MySQLBinlogDataStreamer(MySQLBinlogDataProvider provider, MySQLBinlogState state) {
        this.provider = provider;
        this.processedServerSchema = state.getServerSchema();
        this.processedGtidSet = new GtidSet(state.getGtidSet());
    }

    /**
     * Не потокобезопасно. Вызывать можно только из того же треда, в котором работает стример.
     * Т.е. вызывать getState в консумере - это ок.
     *
     * @return Состояние на момент последней полностью обработанной транзакции.
     */
    public MySQLBinlogState getState() {
        return new MySQLBinlogState(getProcessedServerSchema(), getProcessedGtidSet());
    }

    /**
     * Полезно для использования в консумере, чтобы понимать, нужно ли завершаться graceful.
     *
     * @return true если в процессе работы стримера пришло исключение.
     */
    public boolean hasFailed() {
        return failure != null;
    }

    private synchronized ServerSchema getProcessedServerSchema() {
        return processedServerSchema;
    }

    private synchronized void setProcessedServerSchema(ServerSchema serverSchema) {
        processedServerSchema = serverSchema;
    }

    private synchronized String getProcessedGtidSet() {
        return processedGtidSet != null ? processedGtidSet.toString() : null;
    }

    private synchronized void addProcessedGtid(String gtid) {
        processedGtidSet.add(gtid);
    }

    private void addFailure(Throwable e) {
        try {
            synchronized (this) {
                if (failure == null) {
                    failure = e;
                } else {
                    failure.addSuppressed(e);
                }
            }
        } finally {
            provider.stop();
        }
    }

    private synchronized Throwable takeFailure() {
        Throwable throwable = failure;
        failure = null;
        return throwable;
    }

    private synchronized void startRunning() {
        if (running) {
            throw new MySQLBinlogException("Binlog streaming is already running");
        }
        running = true;
    }

    private synchronized void stopRunning() {
        running = false;
    }

    private BinaryLogClient.LifecycleListener makeLifecycleListener(final MySQLBinlogConsumer consumer) {
        return new BinaryLogClient.LifecycleListener() {
            @Override
            public void onConnect(BinaryLogClient client) {
                try {
                    consumer.onConnect(MySQLBinlogDataStreamer.this);
                } catch (Throwable e) {
                    addFailure(e);
                }
            }

            @Override
            public void onCommunicationFailure(BinaryLogClient client, Exception ex) {
                addFailure(ex);
            }

            @Override
            public void onEventDeserializationFailure(BinaryLogClient client, Exception ex) {
                addFailure(ex);
            }

            @Override
            public void onDisconnect(BinaryLogClient client) {
                try {
                    consumer.onDisconnect(MySQLBinlogDataStreamer.this);
                } catch (Throwable e) {
                    addFailure(e);
                }
            }
        };
    }

    private BinaryLogClient.EventListener makeEventListener(final MySQLBinlogConsumer consumer, final Connection conn) {
        return new BinaryLogClient.EventListener() {
            // NOTE: this code may be running on a separate thread
            private String transactionGtid = null;
            private boolean inTransaction = false;
            private Map<Long, TableMapEventData> tableMaps = new HashMap<>();
            private Map<Long, TableSchema> tableSchemas = new HashMap<>();
            private RowsQueryEventData lastRowsQuery = null;

            private TableSchema findTableSchema(TableMapEventData data) {
                for (DatabaseSchema database : processedServerSchema.getDatabases()) {
                    if (database.getName().equalsIgnoreCase(data.getDatabase())) {
                        for (TableSchema table : database.getTables()) {
                            if (table.getName().equalsIgnoreCase(data.getTable())) {
                                return table;
                            }
                        }
                    }
                }
                return null;
            }

            private TableMapEventData requireTableMap(long tableId) {
                TableMapEventData tableMap = tableMaps.get(tableId);
                if (tableMap == null) {
                    throw new MySQLBinlogException("There is no table map for table with id " + tableId);
                }
                return tableMap;
            }

            private void flushTransaction(String query) {
                if (inTransaction) {
                    throw new MySQLBinlogException(
                            "Found an attempt to commit implicitly while in transaction: " + query);
                }
                if (transactionGtid != null) {
                    addProcessedGtid(transactionGtid);
                    transactionGtid = null;
                }
                tableMaps.clear();
                tableSchemas.clear();
            }

            private void beginTransaction() {
                if (inTransaction) {
                    throw new MySQLBinlogException("Found an attempt to begin transaction while in transaction");
                }
                inTransaction = true;
            }

            private void commitTransaction(String query) {
                if (!inTransaction) {
                    throw new MySQLBinlogException(
                            "Found an attempt to commit transaction while not in transaction: " + query);
                }
                inTransaction = false;
                flushTransaction(query);
            }

            private boolean looksLikeDML(String query) {
                query = MySQLUtils.extractFirstWordFromSQL(query);
                for (String prefix : DML_PREFIXES) {
                    if (prefix.equalsIgnoreCase(query)) {
                        return true;
                    }
                }
                return false;
            }

            @Override
            public void onEvent(Event event) {
                try {
                    switch (event.getHeader().getEventType()) {
                        case FORMAT_DESCRIPTION:
                        case PREVIOUS_GTIDS:
                        case ROTATE:
                            // we are not really interested in these events
                            return;
                        case TABLE_MAP: {
                            TableMapEventData data = event.getData();
                            tableMaps.put(data.getTableId(), data);
                            TableSchema table = findTableSchema(data);
                            if (table != null) {
                                tableSchemas.put(data.getTableId(), table);
                            }
                            return;
                        }
                        case ROWS_QUERY: {
                            RowsQueryEventData data = event.getData();
                            lastRowsQuery = data;
                            consumer.onRowsQuery(data, event.getHeader().getTimestamp());
                            return;
                        }
                        case GTID: {
                            GtidEventData data = event.getData();
                            transactionGtid = data.getGtid();
                            return;
                        }
                        case QUERY: {
                            QueryEventData data = event.getData();
                            String sql = data.getSql();
                            if ("BEGIN".equalsIgnoreCase(sql)) {
                                lastRowsQuery = null;
                                beginTransaction();
                                consumer.onTransactionBegin(transactionGtid);
                            } else if ("COMMIT".equalsIgnoreCase(sql)) {
                                lastRowsQuery = null;
                                String gtid = transactionGtid;
                                commitTransaction(sql);
                                consumer.onTransactionCommit(gtid);
                            } else if ("ROLLBACK".equalsIgnoreCase(sql)) {
                                throw new MySQLBinlogException("Unexpected ROLLBACK found");
                            } else if (sql.startsWith("SAVEPOINT ")) {
                                // We don't care about savepoints
                            } else {
                                // Assume this is a DDL statement
                                if (looksLikeThreadSpecific(event)) {
                                    // ignore thread-specific queries (temporary tables, etc.)
                                    if (!looksLikeDML(sql)) {
                                        flushTransaction("/* THREAD SPECIFIC */ " + sql);
                                    }
                                    return;
                                }
                                EventHeaderV4 header = event.getHeader();
                                if ((header.getFlags() & LOG_EVENT_SUPPRESS_USE_F) == 0
                                        && !StringUtils.isEmpty(data.getDatabase())
                                ) {
                                    try {
                                        conn.setCatalog(data.getDatabase());
                                    } catch (SQLException e) {
                                        throw new MySQLBinlogException("Failed to use the database \n"
                                                + event.toString() + " \n"
                                                + conn.toString(), e);
                                    }
                                }
                                if (!looksLikeDML(sql)) {
                                    ServerSchema before = getProcessedServerSchema();
                                    ServerSchema after = before;
                                    if (BinlogEventConverter.shouldApplySchemaDDL(sql)) {
                                        try (PreparedStatement stmt = conn.prepareStatement(sql)) {
                                            stmt.executeUpdate();
                                        } catch (SQLException e) {
                                            throw new MySQLBinlogException("Failed to apply DDL", e);
                                        }
                                        try {
                                            after = ServerSchema.dump(conn);
                                        } catch (SQLException e) {
                                            throw new MySQLBinlogException("Failed to dump new schema", e);
                                        }
                                        setProcessedServerSchema(after);
                                    }
                                    String gtid = transactionGtid;
                                    flushTransaction(sql);
                                    consumer.onDDL(gtid, data, before, after);
                                }
                            }
                            return;
                        }
                        case XID: {
                            lastRowsQuery = null;
                            String gtid = transactionGtid;
                            commitTransaction("XID COMMIT");
                            consumer.onTransactionCommit(gtid);
                            return;
                        }
                        case WRITE_ROWS:
                        case EXT_WRITE_ROWS: {
                            WriteRowsEventData data = event.getData();
                            TableMapEventData tableMap = requireTableMap(data.getTableId());
                            TableSchema table = tableSchemas.get(data.getTableId());
                            consumer.onInsertRows(new MySQLSimpleData(event.getHeader().getTimestamp(),
                                    data.getRows(), data.getIncludedColumns(), table, tableMap, lastRowsQuery));
                            return;
                        }
                        case UPDATE_ROWS:
                        case EXT_UPDATE_ROWS: {
                            UpdateRowsEventData data = event.getData();
                            TableMapEventData tableMap = requireTableMap(data.getTableId());
                            TableSchema table = tableSchemas.get(data.getTableId());
                            consumer.onUpdateRows(
                                    new MySQLUpdateData(event.getHeader().getTimestamp(), data, table, tableMap,
                                            lastRowsQuery));
                            return;
                        }
                        case DELETE_ROWS:
                        case EXT_DELETE_ROWS: {
                            DeleteRowsEventData data = event.getData();
                            TableMapEventData tableMap = requireTableMap(data.getTableId());
                            TableSchema table = tableSchemas.get(data.getTableId());
                            consumer.onDeleteRows(new MySQLSimpleData(event.getHeader().getTimestamp(),
                                    data.getRows(), data.getIncludedColumns(), table, tableMap, lastRowsQuery));
                            return;
                        }
                    }
                } catch (Throwable e) {
                    addFailure(e);
                }
            }
        };
    }

    public static TmpMySQLServerWithDataDir createMySQL(MySQLServerBuilder builder) throws InterruptedException {
        return TmpMySQLServerWithDataDir.create("server-streamer", builder);
    }

    /**
     * Streams data from provider into the specified consumer
     */
    public void run(final MySQLBinlogConsumer consumer, MySQLInstance mysql) throws IOException, SQLException {
        try (Connection conn = mysql.connect()) {
            // Выставляем наиболее совместимый с неожиданными ALTER'ами режим работы
            MySQLUtils.loosenRestrictions(conn);
            final BinaryLogClient.LifecycleListener lifecycleListener = makeLifecycleListener(consumer);
            final BinaryLogClient.EventListener eventListener = makeEventListener(consumer, conn);
            getProcessedServerSchema().restore(conn);
            try {
                startRunning();
                try {
                    provider.run(getProcessedGtidSet(), lifecycleListener, eventListener);
                } finally {
                    stopRunning();
                }
            } catch (Throwable exc) {
                Throwable failure = takeFailure();
                if (failure != null) {
                    exc.addSuppressed(failure);
                }
                throw exc;
            }
            Throwable failure = takeFailure();
            if (failure != null) {
                if (failure instanceof IOException) {
                    throw (IOException) failure;
                }
                if (failure instanceof RuntimeException) {
                    throw (RuntimeException) failure;
                }
                if (failure instanceof Error) {
                    throw (Error) failure;
                }
                throw new MySQLBinlogException("Unhandled exception", failure);
            }
        }
    }

    /**
     * Asynchronously stops a running run() call
     */
    public void stop() {
        provider.stop();
    }

    @Override
    public String toString() {
        return "MySQLBinlogDataStreamer{" +
                "provider=" + provider +
                '}';
    }
}
