package ru.yandex.direct.mysql;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Objects;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonGetter;
import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonProperty;

import ru.yandex.direct.mysql.schema.ServerSchema;

/**
 * Encapsulates the state of binlog processing
 */
public class MySQLBinlogState {
    private ServerSchema serverSchema;
    private String gtidSet;

    @JsonCreator
    public MySQLBinlogState(@JsonProperty("server_schema") ServerSchema serverSchema,
                            @JsonProperty("gtid_set") String gtidSet) {
        this.serverSchema = Objects.requireNonNull(serverSchema);
        this.gtidSet = gtidSet;
    }

    public MySQLBinlogState(byte[] serializedServer, String gtidSet) {
        this(ServerSchema.fromJson(serializedServer), gtidSet);
    }

    @JsonGetter("server_schema")
    public ServerSchema getServerSchema() {
        return serverSchema;
    }

    @JsonGetter("gtid_set")
    public String getGtidSet() {
        return gtidSet;
    }

    @JsonIgnore
    public byte[] getSerializedServerSchema() {
        return serverSchema.toJsonBytes();
    }

    public static String getGtidExecuted(Connection conn) throws SQLException {
        String gtidSet = null;
        try (PreparedStatement stmt = conn.prepareStatement("SELECT @@GLOBAL.GTID_EXECUTED")) {
            try (ResultSet rs = stmt.executeQuery()) {
                if (rs.next()) {
                    gtidSet = rs.getString(1);
                }
            }
        }
        return gtidSet;
    }

    /**
     * <p>Создаёт snapshot состояния базы по соединению conn.</p>
     * <p>После выполнения метода в conn может быть установлен другой каталог, поэтому перед вызовом необходимо
     * запомнить текущий каталог, а после вызова вернуть его.</p>
     *
     * @param conn
     * @return состояние базы
     * @throws SQLException
     */
    public static MySQLBinlogState snapshot(Connection conn) throws SQLException {
        // Порядок важен, сначала читаем gtid set, потом делаем дамп
        String gtidExecuted = getGtidExecuted(conn);
        ServerSchema serverSchema = ServerSchema.dump(conn);
        return new MySQLBinlogState(serverSchema, gtidExecuted);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }

        MySQLBinlogState state = (MySQLBinlogState) o;

        if (getServerSchema() != null ? !getServerSchema().equals(state.getServerSchema()) :
                state.getServerSchema() != null) {
            return false;
        }
        return getGtidSet() != null ? getGtidSet().equals(state.getGtidSet()) : state.getGtidSet() == null;

    }

    @Override
    public int hashCode() {
        int result = getServerSchema() != null ? getServerSchema().hashCode() : 0;
        result = 31 * result + (getGtidSet() != null ? getGtidSet().hashCode() : 0);
        return result;
    }
}
