package ru.yandex.direct.mysql;

import java.io.Serializable;
import java.util.AbstractList;
import java.util.Arrays;
import java.util.List;

import ru.yandex.direct.mysql.schema.ColumnSchema;

public class MySQLSimpleRow extends AbstractList<MySQLColumnData> {
    private final List<ColumnSchema> columns;
    private final Serializable[] values;
    private final List<ColumnSchema> defaultColumns;
    private final int size;

    public MySQLSimpleRow(List<ColumnSchema> columns, Serializable[] values, List<ColumnSchema> defaultColumns) {
        if (columns.size() != values.length) {
            throw new IllegalArgumentException(
                    "Number of columns (" + columns.size() + ") does not match " +
                            "number of values (" + values.length + ")"
            );
        }
        this.columns = columns;
        this.values = values;
        this.defaultColumns = defaultColumns;
        this.size = defaultColumns != null ? values.length + defaultColumns.size() : values.length;
    }

    public MySQLSimpleRow(List<ColumnSchema> columns, Serializable[] values) {
        this(columns, values, null);
    }

    @Override
    public MySQLColumnData get(int index) {
        if (index < values.length) {
            return new MySQLColumnData(columns.get(index), values[index]);
        }
        if (defaultColumns != null && index < size) {
            ColumnSchema defaultColumn = defaultColumns.get(index - values.length);
            MySQLColumnType columnType = MySQLColumnType.getCached(defaultColumn.getColumnType());
            Serializable value = columnType.getCachedDefaultValue(defaultColumn.getDefaultValue());
            return new MySQLColumnData(defaultColumn, value);
        }
        throw new IndexOutOfBoundsException("Index: " + index + ", Size: " + size);
    }

    public ColumnSchema getSchema(int index) {
        if (index < columns.size()) {
            return columns.get(index);
        } else if (index < size) {
            return defaultColumns.get(index - columns.size());
        } else {
            throw new IndexOutOfBoundsException("Index: " + index + ", Size: " + size);
        }
    }

    @Override
    public int size() {
        return size;
    }

    @Override
    public String toString() {
        return "MySQLSimpleRow{" +
                "columns=" + columns +
                ", values=" + Arrays.toString(values) +
                ", defaultColumns=" + defaultColumns +
                ", size=" + size +
                '}';
    }
}
