package ru.yandex.direct.mysql;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.BitSet;
import java.util.List;
import java.util.Map;

import com.github.shyiko.mysql.binlog.event.RowsQueryEventData;
import com.github.shyiko.mysql.binlog.event.TableMapEventData;
import com.github.shyiko.mysql.binlog.event.UpdateRowsEventData;

import ru.yandex.direct.mysql.schema.ColumnSchema;
import ru.yandex.direct.mysql.schema.TableSchema;

public class MySQLUpdateData {
    private long timestamp;
    private TableSchema schema;
    private TableMapEventData tableMap;
    private RowsQueryEventData rowsQuery;
    private List<Map.Entry<Serializable[], Serializable[]>> rows;
    private List<ColumnSchema> beforeColumns;
    private List<ColumnSchema> afterColumns;

    public MySQLUpdateData(long timestamp, UpdateRowsEventData data, TableSchema schema, TableMapEventData tableMap,
                           RowsQueryEventData rowsQuery) {
        this.timestamp = timestamp;
        this.schema = schema;
        this.tableMap = tableMap;
        this.rowsQuery = rowsQuery;
        rows = data.getRows();
        BitSet includedBeforeColumns = data.getIncludedColumnsBeforeUpdate();
        BitSet includedAfterColumns = data.getIncludedColumns();
        byte[] columnTypes = tableMap.getColumnTypes();
        if (schema != null && schema.getColumns().size() != columnTypes.length) {
            throw new MySQLBinlogException(
                    "Table " + tableMap.getDatabase() + "." + tableMap.getTable() + " has schema with " + schema
                            .getColumns().size() + " columns, but binlog data has " + columnTypes.length + " columns");
        }
        beforeColumns = new ArrayList<>();
        afterColumns = new ArrayList<>();
        for (int i = 0; i < columnTypes.length; ++i) {
            if (includedBeforeColumns.get(i)) {
                beforeColumns.add(getColumn(i));
            }
            if (includedAfterColumns.get(i)) {
                afterColumns.add(getColumn(i));
            }
        }
    }

    private ColumnSchema getColumn(int index) {
        if (schema != null) {
            return schema.getColumns().get(index);
        }
        // Table schema is unknown, generate a fake column schema
        return new ColumnSchema("unknown_column_" + (index + 1), "any", "any", null, true);
    }

    public long getTimestamp() {
        return timestamp;
    }

    public TableSchema getSchema() {
        return schema;
    }

    public TableMapEventData getTableMap() {
        return tableMap;
    }

    public String getRowsQuery() {
        return rowsQuery != null ? rowsQuery.getQuery() : "";
    }

    public MySQLUpdateRows getRows() {
        return new MySQLUpdateRows(beforeColumns, afterColumns, rows);
    }

    public List<ColumnSchema> getBeforeColumns() {
        return beforeColumns;
    }
}
