package ru.yandex.direct.mysql;

import java.io.IOException;
import java.nio.file.Path;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.time.Duration;
import java.util.Arrays;
import java.util.List;

import org.apache.commons.io.FileUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.utils.Checked;
import ru.yandex.direct.utils.Transient;
import ru.yandex.direct.utils.db.MySQLConnector;
import ru.yandex.direct.utils.io.TempDirectory;

public class TmpMySQLServerWithDataDir implements MySQLInstance {
    private static final Logger logger = LoggerFactory.getLogger(TmpMySQLServerWithDataDir.class);

    private static final MysqlBinlogRowImage DEFAULT_BINLOG_ROW_IMAGE = MysqlBinlogRowImage.NOBLOB;

    private static final List<String> BINLOG_PARAMS = Arrays.asList(
            "--binlog-format=row",
            "--binlog-rows-query-log-events",
            "--enforce-gtid-consistency",
            "--gtid-mode=ON",
            "--log-bin=binlog",
            "--log-slave-updates",
            "--master-info-repository=TABLE"
    );

    private MySQLServer server;
    private TempDirectory tempDirectory;

    public TmpMySQLServerWithDataDir(MySQLServer server, TempDirectory tempDirectory) {
        logger.info("started mysqld on {}", tempDirectory);
        this.server = server;
        this.tempDirectory = tempDirectory;
    }

    public static TmpMySQLServerWithDataDir create(String name, MySQLServerBuilder builder)
            throws InterruptedException {
        try (Transient<TempDirectory> tmpDir = new Transient<>()) {
            tmpDir.item = builder.createTempDirectory(name);
            builder.setDataAndConfigDir(tmpDir.item.getPath());
            builder.initializeDataDir();
            try (Transient<TmpMySQLServerWithDataDir> mysql = new Transient<>()) {
                mysql.item = new TmpMySQLServerWithDataDir(builder.start(), tmpDir.item);
                tmpDir.success();
                return mysql.pop();
            }
        }
    }

    public static TmpMySQLServerWithDataDir create(String name, MySQLServerBuilder builder, Path dataDirTemplate) {
        try (Transient<TempDirectory> tmpDir = new Transient<>()) {
            tmpDir.item = builder.createTempDirectory(name);
            builder.setDataAndConfigDir(tmpDir.item.getPath());

            try {
                FileUtils.copyDirectory(dataDirTemplate.toFile(), tmpDir.item.getPath().toFile());
            } catch (IOException exc) {
                throw new Checked.CheckedException(exc);
            }

            try (Transient<TmpMySQLServerWithDataDir> mysql = new Transient<>()) {
                mysql.item = new TmpMySQLServerWithDataDir(builder.start(), tmpDir.item);
                tmpDir.success();
                return mysql.pop();
            }
        }
    }

    public static TmpMySQLServerWithDataDir createWithBinlog(String name, MySQLServerBuilder builder)
            throws InterruptedException {
        return create(name, setupBinLog(builder));
    }

    public static TmpMySQLServerWithDataDir createWithBinlog(
            String name, MySQLServerBuilder builder, MysqlBinlogRowImage rowImage
    )
            throws InterruptedException {
        return create(name, setupBinLog(builder, rowImage));
    }

    public static TmpMySQLServerWithDataDir createWithBinlog(String name, MySQLServerBuilder builder,
                                                             Path dataDirTemplate) {
        return create(name, setupBinLog(builder), dataDirTemplate);
    }

    // * Requires mysql 5.7.6+, on OSX may be installed with brew install percona-server
    public static MySQLServerBuilder setupBinLog(MySQLServerBuilder builder) {
        return setupBinLog(builder, DEFAULT_BINLOG_ROW_IMAGE);
    }

    // * Requires mysql 5.7.6+, on OSX may be installed with brew install percona-server
    public static MySQLServerBuilder setupBinLog(MySQLServerBuilder builder,  MysqlBinlogRowImage rowImage) {
        if (builder.getServerId() == 0) {
            throw new IllegalArgumentException("Server ID must be set to non-zero to use binlog");
        }
        return builder
                .addExtraArgs(BINLOG_PARAMS)
                .addExtraArgs("--binlog-row-image=" + rowImage)
                ;
    }

    public static TmpMySQLServerWithDataDir createSlave(String name, MySQLServerBuilder builder, MySQLInstance master)
            throws InterruptedException {
        try (Transient<TmpMySQLServerWithDataDir> slave = new Transient<>()) {
            slave.item = createWithBinlog(name, builder);
            setupSlave(slave.item, master);
            return slave.pop();
        }
    }

    public static TmpMySQLServerWithDataDir createSlave(String name, MySQLServerBuilder builder, MySQLInstance master,
                                                        Path dataDirTemplate) {
        try (Transient<TmpMySQLServerWithDataDir> slave = new Transient<>()) {
            slave.item = createWithBinlog(name, builder, dataDirTemplate);
            setupSlave(slave.item, master);
            return slave.pop();
        }
    }

    public static void setupSlave(MySQLConnector slave, MySQLConnector master) {
        try (Connection conn = slave.connect()) {
            setupSlave(conn, master);
        } catch (SQLException exc) {
            throw new Checked.CheckedException(exc);
        }
    }

    @SuppressWarnings("squid:S2068") // ложноположительное срабатывание на указание plain-text пароля в запросе
    public static void setupSlave(Connection slave, MySQLConnector master) {
        try {
            try (PreparedStatement statement = slave.prepareStatement(
                    "CHANGE MASTER TO "
                            + "master_host=?, "
                            + "master_port=?, "
                            + "master_user=?, "
                            + "master_password=?, "
                            + "master_auto_position=1"
            )) {
                statement.setString(1, master.getHost());
                statement.setInt(2, master.getPort());
                statement.setString(3, master.getUsername());
                statement.setString(4, master.getPassword());
                statement.execute();
            }
            MySQLUtils.executeUpdate(slave, "START SLAVE");
        } catch (SQLException exc) {
            throw new Checked.CheckedException(exc);
        }
    }

    @Override
    public String getHost() {
        return server.getHost();
    }

    @Override
    public int getPort() {
        return server.getPort();
    }

    @Override
    public String getUsername() {
        return server.getUsername();
    }

    @Override
    public String getPassword() {
        return server.getPassword();
    }

    @Override
    public Connection connect() throws SQLException {
        return server.connect();
    }

    @Override
    public Connection connect(Duration timeout) throws SQLException {
        return server.connect(timeout);
    }

    @Override
    public void close() {
        try {
            server.close();
        } finally {
            tempDirectory.close();
        }
    }

    @Override
    public void awaitConnectivity(Duration timeout) {
        server.awaitConnectivity(timeout);
    }
}
