package ru.yandex.direct.mysql.ytsync.export.components;

import java.sql.Connection;
import java.sql.SQLException;
import java.util.Deque;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedDeque;
import java.util.concurrent.ConcurrentMap;

import javax.annotation.PreDestroy;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Component;

import ru.yandex.direct.mysql.ytsync.common.util.MySqlConnectionProvider;

/**
 * Компонент, добавляющий кеширование к соединениям из ImportConnectionProvider
 */
@Lazy
@Component
public class ConnectionsCache {
    private static final Logger logger = LoggerFactory.getLogger(ConnectionsCache.class);

    private final MySqlConnectionProvider exportConnectionProvider;
    private final ConcurrentMap<String, Deque<Connection>> connectionsCache = new ConcurrentHashMap<>();

    @Autowired
    public ConnectionsCache(MySqlConnectionProvider exportConnectionProvider) {
        this.exportConnectionProvider = exportConnectionProvider;
    }

    /**
     * Возвращает очередь соединений для dbName
     */
    private Deque<Connection> getQueue(String dbName) {
        Deque<Connection> connections = connectionsCache.get(dbName);
        if (connections == null) {
            connections = connectionsCache.computeIfAbsent(dbName, ignored -> new ConcurrentLinkedDeque<>());
        }
        return connections;
    }

    /**
     * Возвращает соединение для базы dbName
     */
    public Connection getConnection(String dbName) throws SQLException {
        Deque<Connection> connections = getQueue(dbName);
        Connection conn;
        while (true) {
            conn = connections.poll();
            if (conn == null) {
                conn = exportConnectionProvider.createConnectionForImport(dbName);
                break;
            }
            // Возвращаем соединение только если оно живое
            if (conn.isValid(5)) {
                break;
            }
        }
        return conn;
    }

    /**
     * Отдаёт соединение conn назад в пул для dbName
     */
    public void releaseConnection(String dbName, Connection conn) {
        Deque<Connection> connections = getQueue(dbName);
        connections.addFirst(conn);
    }

    /**
     * Закрывает все закешированные соединения
     */
    @PreDestroy
    public void closeCached() {
        connectionsCache.forEach((dbName, connections) -> {
            Connection conn;
            while ((conn = connections.poll()) != null) {
                try {
                    conn.close();
                } catch (SQLException e) {
                    logger.warn("Failed to close cached " + dbName + " connection", e);
                }
            }
        });
    }
}
