package ru.yandex.direct.mysql.ytsync.export.components;

import java.time.Duration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMap;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Component;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.IteratorF;
import ru.yandex.direct.mysql.ytsync.common.keys.PivotKeys;
import ru.yandex.direct.mysql.ytsync.export.task.ExportConfig;
import ru.yandex.direct.ytwrapper.YtUtils;
import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.inside.yt.kosher.common.DataSize;
import ru.yandex.inside.yt.kosher.common.GUID;
import ru.yandex.inside.yt.kosher.cypress.CypressNodeType;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.impl.operations.utils.ReducerWithKey;
import ru.yandex.inside.yt.kosher.impl.transactions.utils.YtTransactionsUtils;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTree;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTreeBuilder;
import ru.yandex.inside.yt.kosher.impl.ytree.object.annotation.YTreeObject;
import ru.yandex.inside.yt.kosher.operations.Statistics;
import ru.yandex.inside.yt.kosher.operations.Yield;
import ru.yandex.inside.yt.kosher.operations.specs.JobIo;
import ru.yandex.inside.yt.kosher.operations.specs.ReduceSpec;
import ru.yandex.inside.yt.kosher.operations.specs.ReducerSpec;
import ru.yandex.inside.yt.kosher.operations.specs.SortSpec;
import ru.yandex.inside.yt.kosher.tables.TableWriterOptions;
import ru.yandex.inside.yt.kosher.ytree.YTreeMapNode;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;
import ru.yandex.yt.ytclient.tables.ColumnSchema;
import ru.yandex.yt.ytclient.tables.TableSchema;

import static ru.yandex.direct.mysql.ytsync.common.util.YtSyncCommonUtil.YTSYNC_LOGGER;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Lazy
@Component
@ParametersAreNonnullByDefault
public class StaticToDynamicConverter {
    private final Yt yt;
    private final ExportConfig exportConfig;

    @Autowired
    public StaticToDynamicConverter(Yt yt, ExportConfig exportConfig) {
        this.yt = yt;
        this.exportConfig = exportConfig;
    }

    private List<String> extractKeyNames(TableSchema tableSchema) {
        // def extract_key_names(schema):
        //    """Возвращает список названий ключевых колонок"""
        //    key_names = []
        //    for column in schema:
        //        if column.get('sort_order'):
        //            key_names.append(column['name'])
        //    return key_names
        return tableSchema.getColumns().stream()
                .filter(c -> c.getSortOrder() != null)
                .map(ColumnSchema::getName)
                .collect(Collectors.toList());
    }

    @YTreeObject
    public static class KeepFirstRowReducer implements ReducerWithKey<YTreeMapNode, YTreeMapNode, String> {
        private final List<String> keys;

        public KeepFirstRowReducer(List<String> keys) {
            this.keys = keys;
        }
        //class ReduceKeepFirstRow(object):
        //    """Оставляет только первую строку среди дубликатов"""
        //
        //    def __call__(self, key, rows):
        //        for row in rows:
        //            yield row
        //            break

        @Override
        public String key(YTreeMapNode entry) {
            StringBuilder sb = new StringBuilder();
            keys.forEach(k -> sb.append(entry.getOrThrow(k)));
            return sb.toString();
        }

        @Override
        public void reduce(String s, IteratorF<YTreeMapNode> entries, Yield<YTreeMapNode> yield,
                           Statistics statistics) {
            if (entries.hasNext()) {
                yield.yield(entries.next());
            }
        }
    }

    private YPath getSortedPath(YPath path) {
        return YPath.simple(path.toString() + ".sorted");
    }

    public boolean sortedTableExists(YPath target, @Nullable GUID transactionUID) {
        return yt.cypress().exists(Optional.ofNullable(transactionUID), transactionUID != null, getSortedPath(target));
    }

    public void convert(List<YPath> sources, YPath target, TableSchema schema, @Nullable PivotKeys pivotKeys,
                        boolean deleteSources) {
        YPath sortedTable = getSortedPath(target);
        if (!sortedTableExists(target, null)) {
            YtTransactionsUtils.withTransaction(
                    yt,
                    Duration.ofMinutes(1),
                    Optional.of(Duration.ofSeconds(15)),
                    transaction -> {
                        GUID transactionUID = transaction.getId();
                        YTSYNC_LOGGER.info("Run with YT-transaction {}", transactionUID);
                        try {
                            sortInsideTransaction(transactionUID, sources, sortedTable, schema, deleteSources);
                        } finally {
                            YTSYNC_LOGGER.info("Closing YT-transaction {}", transactionUID);
                        }
                        return true;
                    });
        }

        YtTransactionsUtils.withTransaction(
                yt,
                Duration.ofMinutes(1),
                Optional.of(Duration.ofSeconds(15)),
                transaction -> {
                    GUID transactionUID = transaction.getId();
                    YTSYNC_LOGGER.info("Run with YT-transaction {}", transactionUID);
                    try {
                        reduceInsideTransaction(transactionUID, sortedTable, target, schema, pivotKeys);
                    } finally {
                        YTSYNC_LOGGER.info("Closing YT-transaction {}", transactionUID);
                    }
                    return true;
                });

        // Делаем таблицу динамической
        yt.tables().alterTable(target, Optional.of(true), Optional.empty());

        if (exportConfig.profilingByTag()) {
            yt.cypress().set(
                    target.attribute("profiling_mode"),
                    YTree.node("tag")
            );
            yt.cypress().set(
                    target.attribute("profiling_tag"),
                    "direct/mysql-sync/" + exportConfig.version() + "/" + target.name()
            );
        }

        if (exportConfig.dynamicStoreRead()) {
            yt.cypress().set(
                    target.attribute("enable_dynamic_store_read"),
                    YTree.booleanNode(true)
            );
        }

        // Перешардируем таблицу, если указаны соответствующие ключи
        if (pivotKeys != null) {
            List<List<YTreeNode>> listF =
                    pivotKeys.getKeys().stream().map(YTreeNode::asList).collect(Collectors.toList());
            yt.tables().reshard(target, listF);
        }
    }

    private YTreeNode getYTreeSchema(boolean uniqueKeys, TableSchema tableSchema) {
        YTreeBuilder schemaBuilder = YTree.builder();

        schemaBuilder.beginAttributes();
        schemaBuilder
                .key("unique_keys")
                .value(YTree.booleanNode(uniqueKeys));
        schemaBuilder.endAttributes();

        schemaBuilder.beginList();
        for (ColumnSchema columnSchema : tableSchema.getColumns()) {
            YTreeBuilder fieldBuilder = YTree.mapBuilder()
                    .key("name").value(columnSchema.getName())
                    .key("type").value(columnSchema.getType().getName());
            if (columnSchema.getSortOrder() != null) {
                fieldBuilder.key("sort_order").value(columnSchema.getSortOrder().getName());
            }
            if (columnSchema.getExpression() != null) {
                fieldBuilder.key("expression").value(columnSchema.getExpression());
            }
            schemaBuilder.value(fieldBuilder.buildMap());
        }
        schemaBuilder.endList();

        return schemaBuilder.build();
    }

    private void sortInsideTransaction(GUID transactionUID, List<YPath> sources, YPath sortedTable, TableSchema schema,
                                       boolean deleteSources) {
        //    # Схема, в которой будут промежуточные данные
        //    key_names = extract_key_names(args.schema)
        //    column_names = extract_column_names(args.schema)

        //        sorted_attrs = dict(
        //            schema=make_non_unique_schema(args.schema),
        //            optimize_for=args.optimize_for,
        //            primary_medium=args.primary_medium,
        //        )
        Map<String, YTreeNode> createAttrs = new HashMap<>();
        createAttrs.put(YtUtils.SCHEMA_ATTR, getYTreeSchema(false, schema));
        createAttrs.put("optimize_for", YTree.stringNode(exportConfig.getOptimizeFor().getText()));
        createAttrs.put("primary_medium", YTree.stringNode(exportConfig.ytTablesMedium()));

        //        yt.create_table(
        //            sorted_table,
        //            recursive=True,
        //            attributes=sorted_attrs,
        //        )
        yt.cypress()
                .create(Optional.of(transactionUID), true, sortedTable, CypressNodeType.TABLE, true, false, createAttrs);

        //        yt.run_sort(
        //            source_table=[yt.TablePath(t, columns=column_names) for t in args.inputs],
        //            destination_table=sorted_table,
        //            sort_by=key_names,
        //            spec=dict(
        //                intermediate_data_account=args.intermediate_account,
        //                intermediate_data_medium=args.primary_medium,
        //            ),
        //        )
        SortSpec sortSpec = SortSpec.builder()
                .setInputTables(mapList(sources, yp -> yp.withColumns(Cf.wrap(schema.getColumnNames()))))
                .setOutputTable(sortedTable)
                .setSortBy(extractKeyNames(schema))
                .setAdditionalSpecParameters(ImmutableMap.of(
                        // Флаг, из-за отсутствия которого падали джобы (см YTADMINREQ-17641)
                        "enable_partitioned_data_balancing", YTree.booleanNode(false),
                        // Нужно использовать свой аккаунт, иначе операция может упасть
                        // (В zeno уже не влезает см YTADMINREQ-20549)
                        "intermediate_data_account", YTree.stringNode(YtUtils.getYtAccount(yt,
                                YPath.simple(exportConfig.rootPath())))
                ))
                .build();
        yt.operations().sortAndGetOp(Optional.of(transactionUID), true, sortSpec)
                .awaitAndThrowIfNotSuccess();

        if (deleteSources) {
            // Отсортированная таблица теперь существует, удаляем исходные статические
            for (YPath sourceTableName : sources) {
                if (yt.cypress().exists(Optional.of(transactionUID), true, sourceTableName)) {
                    yt.cypress().remove(Optional.of(transactionUID), true, sourceTableName);
                }
            }
        }
    }

    private void reduceInsideTransaction(GUID transactionUID, YPath sortedTable, YPath target, TableSchema schema,
                                         @Nullable PivotKeys pivotKeys) {
        //        if yt.exists(args.output):
        //            yt.remove(args.output)
        if (yt.cypress().exists(Optional.of(transactionUID), true, target)) {
            yt.cypress().remove(Optional.of(transactionUID), true, target);
        }

        //        reduced_attrs = dict(
        //            schema=args.schema,
        //            external=False,
        //            optimize_for=args.optimize_for,
        //            primary_medium=args.primary_medium,
        //        )
        //        if args.pivot_keys is not None:
        //            # Отключаем балансировщик, т.к. мы будем шардировать вручную
        //            reduced_attrs['enable_tablet_balancer'] = False
        //            reduced_attrs['disable_tablet_balancer'] = True
        Map<String, YTreeNode> createReducedAttrs = new HashMap<>();
        createReducedAttrs.put(YtUtils.SCHEMA_ATTR, getYTreeSchema(true, schema));
        createReducedAttrs.put("optimize_for", YTree.stringNode(exportConfig.getOptimizeFor().getText()));
        createReducedAttrs.put("primary_medium", YTree.stringNode(exportConfig.ytTablesMedium()));
        if (pivotKeys != null) {
            createReducedAttrs.put("enable_tablet_balancer", YTree.booleanNode(false));
            createReducedAttrs.put("disable_tablet_balancer", YTree.booleanNode(true));
        }

        //        yt.create_table(
        //            args.output,
        //            recursive=True,
        //            attributes=reduced_attrs,
        //        )
        yt.cypress()
                .create(Optional.of(transactionUID), true, target, CypressNodeType.TABLE, true, false,
                        createReducedAttrs);

        //        yt.run_reduce(
        //            ReduceKeepFirstRow(),
        //            source_table=sorted_table,
        //            destination_table=yt.TablePath(args.output, sorted_by=key_names),
        //            format=YSON_FORMAT,
        //            sort_by=key_names,
        //            reduce_by=key_names,
        //            spec=dict(
        //                intermediate_data_account=args.intermediate_account,
        //                intermediate_data_medium=args.primary_medium,
        //                data_size_per_job=512 * MB,
        //                job_io=dict(
        //                    table_witer=dict(
        //                        block_size=256 * KB,
        //                        desired_chunk_size=100 * MB,
        //                    ),
        //                ),
        //            ),
        //        )
        ReduceSpec reduceSpec = ReduceSpec.builder()
                .setInputTables(Cf.list(sortedTable))
                .setOutputTables(Cf.list(target))
                .setReduceBy(Cf.wrap(extractKeyNames(schema)))
                .setReducerSpec(ReducerSpec.builder(new KeepFirstRowReducer(extractKeyNames(schema))).build())
                .setDataSizePerJob(DataSize.fromMegaBytes(512))
                .setJobIo(new JobIo(new TableWriterOptions()
                        .withBlockSize(DataSize.fromKiloBytes(256))
                        .withDesiredChunkSize(DataSize.fromMegaBytes(100))))
                .build();
        yt.operations().reduceAndGetOp(Optional.of(transactionUID), true, reduceSpec)
                .awaitAndThrowIfNotSuccess();

        // Удаляем временную сортированную таблицу
        yt.cypress().remove(Optional.of(transactionUID), true, sortedTable);
    }
}
