package ru.yandex.direct.mysql.ytsync.export.components;

import java.sql.Connection;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.function.BooleanSupplier;

import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.PreDestroy;

import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Component;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.direct.mysql.MySQLBinlogState;
import ru.yandex.direct.mysql.ytsync.common.compatibility.KosherCompressorFactory;
import ru.yandex.direct.mysql.ytsync.common.compatibility.YtLockUtil;
import ru.yandex.direct.mysql.ytsync.common.compatibility.YtSupport;
import ru.yandex.direct.mysql.ytsync.common.components.SyncStatesTable;
import ru.yandex.direct.mysql.ytsync.common.components.SyncStatesTableFactory;
import ru.yandex.direct.mysql.ytsync.common.util.YtSyncCommonUtil;
import ru.yandex.direct.mysql.ytsync.export.task.ExportConfig;
import ru.yandex.direct.mysql.ytsync.export.task.TableExportTask;
import ru.yandex.direct.mysql.ytsync.export.task.TableExportTemplate;
import ru.yandex.direct.mysql.ytsync.export.util.ShardedTableImporter;
import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.inside.yt.kosher.cypress.CypressNodeType;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTree;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.yt.ytclient.tables.TableSchema;

import static ru.yandex.direct.mysql.ytsync.common.util.YtSyncCommonUtil.waitAll;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Позволяет сделать начальный импорт данных
 */
@Lazy
@Component
@ParametersAreNonnullByDefault
public class TablesExporter {
    private static final Logger logger = LoggerFactory.getLogger(TablesExporter.class);

    private final SyncStatesTableFactory statesTableFactory;
    private final Yt yt;
    private final YtSupport ytSupport;
    private final StaticToDynamicConverter staticToDynamicConverter;
    private final ExportConfig exportConfig;
    private final ConnectionsCache connectionsCache;
    private final KosherCompressorFactory compressorFactory;
    private final Map<Pair<String, String>, TableExportTask> tableNameAndPathToTemplate;
    private final List<TableExportTemplate> tableExportTemplates;
    private final List<TableExportTemplate> tableIndexExportTemplates;
    private volatile ShardedTableImporter currentImporter = null;
    private volatile InitialImportMetrics metricsState = null;

    @Autowired
    public TablesExporter(
            SyncStatesTableFactory statesTableFactory, Yt yt,
            YtSupport ytSupport,
            ExportConfig exportConfig,
            ConnectionsCache connectionsCache,
            KosherCompressorFactory compressorFactory,
            Map<Pair<String, String>, TableExportTask> tableNameAndPathToTemplate,
            StaticToDynamicConverter staticToDynamicConverter,
            List<TableExportTemplate> tableExportTemplates, List<TableExportTemplate> tableIndexExportTemplates) {
        this.statesTableFactory = statesTableFactory;
        this.yt = yt;
        this.ytSupport = ytSupport;
        this.exportConfig = exportConfig;
        this.connectionsCache = connectionsCache;
        this.compressorFactory = compressorFactory;
        this.tableNameAndPathToTemplate = tableNameAndPathToTemplate;
        this.tableIndexExportTemplates = tableIndexExportTemplates;
        this.tableExportTemplates = tableExportTemplates;
        this.staticToDynamicConverter = staticToDynamicConverter;
    }

    private YPath getInitialImportFinishedPath() {
        if (exportConfig.importAllTables()) {
            String dbNames = YtSyncCommonUtil.getAllTablesImportInitialSubDir(exportConfig.getDbNames());
            return YPath.simple(exportConfig.rootPath()).child(dbNames).attribute("initial-import-finished");
        } else {
            return YPath.simple(exportConfig.rootPath()).attribute("initial-import-finished");
        }
    }

    private boolean isInitialImportFinished() {
        YPath initialImportFinishedPath = getInitialImportFinishedPath();
        return yt.cypress().exists(initialImportFinishedPath) &&
                yt.cypress().get(initialImportFinishedPath).boolValue();
    }

    private void setInitialImportFinished(boolean finished) {
        YPath initialImportFinishedPath = getInitialImportFinishedPath();
        yt.cypress().set(initialImportFinishedPath, YTree.booleanNode(finished));
    }

    private YPath getInitialImportLockPath() {
        if (exportConfig.importAllTables()) {
            String dbNames = YtSyncCommonUtil.getAllTablesImportInitialSubDir(exportConfig.getDbNames());
            return YPath.simple(exportConfig.rootPath()).child(dbNames).child("initial-import-lock");
        } else {
            return YPath.simple(exportConfig.rootPath()).child("initial-import-lock");
        }
    }

    public void requireInitialImport() {
        // Проверяем, что начальный импорт помечен завершённым
        if (!isInitialImportFinished()) {
            throw new IllegalStateException("Initial import is required");
        }
    }

    public void verifyInitialImport() {
        requireInitialImport();

        // Проверяем правильность таблицы sync-states
        SyncStatesTable syncStatesTable = statesTableFactory.getSyncStatesTable();
        syncStatesTable.verifyAndMount();

        // Проверяем, что все синхронизируемые базы помечены как импортированные
        for (String dbName : exportConfig.getDbNames()) {
            if (!syncStatesTable.isImported(dbName)) {
                throw new IllegalStateException("Database " + dbName + " is not marked as imported");
            }
        }

        ExecutorService executorService = Executors.newFixedThreadPool(10);
        try {
            List<Pair<String, TableSchema>> tasks = new ArrayList<>();

            // Проверяем правильность всех импортируемых таблиц
            for (TableExportTemplate dynamicTable : tableExportTemplates) {
                String path = dynamicTable.getTargetTable();
                tasks.add(Pair.of(path, dynamicTable.getSchema()));
            }
            // Проверяем правильность всех индексов
            for (TableExportTemplate dynamicIndex : tableIndexExportTemplates) {
                String path = dynamicIndex.getTargetTable();
                tasks.add(Pair.of(path, dynamicIndex.getSchema()));
            }
            //
            logger.info("Starting verifying dynamic tables..");
            waitAll(beginVerifyDynamicTables(tasks, executorService));
            logger.info("Finished verifying dynamic tables");
            //
        } finally {
            // Без остановки этого пула программа не сможет завершиться
            logger.info("Stopping verifyTables executorService..");
            executorService.shutdownNow();
            logger.info("Stopped verifyTables executorService");
        }

        metricsState = new InitialImportMetrics(tableExportTemplates.size(), tableExportTemplates.size());
    }

    private List<Future<?>> beginVerifyDynamicTables(List<Pair<String, TableSchema>> tasks, ExecutorService executor) {
        List<Future<?>> futures = new ArrayList<>();
        for (var task : tasks) {
            futures.add(executor.submit(() -> {
                YtSyncCommonUtil.verifyDynamicTable(ytSupport, task.getLeft(), task.getRight());
            }));
        }
        return futures;
    }

    public void runInitialImport() {
        if (isInitialImportFinished()) {
            // Начальный импорт уже завершён, просто выходим
            return;
        }

        YPath initialImportLockPath = getInitialImportLockPath();
        if (!yt.cypress().exists(initialImportLockPath)) {
            yt.cypress().create(initialImportLockPath, CypressNodeType.BOOLEAN, true, true);
        }

        BooleanSupplier interruptedSupplier = () -> Thread.currentThread().isInterrupted();
        YtLockUtil.runInLock(yt, initialImportLockPath, interruptedSupplier, transaction -> {
            if (isInitialImportFinished()) {
                // Другой процесс уже завершил импорт и пометил его выполненным
                logger.info("Import finished by another process");
                return;
            }

            // Запускаем начальный импорт
            runInitialImportUnsafe();

            // На всякий случай проверяем, что наш поток не прерван
            checkInterrupted();

            // Помечаем начальный импорт выполненным, чтобы больше даже не пытаться его запускать
            setInitialImportFinished(true);
        });
    }

    private void runInitialImportUnsafe() throws SQLException, InterruptedException {
        InitialImportState state = new InitialImportState(statesTableFactory.getSyncStatesTable());

        // Подготавливаем таблицу состояний
        state.prepareTable();

        // Проверяем необходимость импорта баз
        for (String dbName : exportConfig.getDbNames()) {
            checkDatabase(state, dbName);
        }

        if (state.importedCount > 0 && state.unimportedCount > 0) {
            throw new IllegalStateException(
                    "Inconsistent state: " + state.importedCount + " databases imported, " + state.unimportedCount
                            + " databases not imported");
        }

        // Запоминаем все новые снепшоты, в случае рестартов мы не будем делать их повторно
        checkInterrupted();
        state.commit(ytSupport);

        if (state.unimportedCount == 0) {
            // Нам нечего импортировать
            return;
        }

        int importedTablesCount = 0;
        for (TableExportTemplate dynamicTable : tableExportTemplates) {
            metricsState = new InitialImportMetrics(tableExportTemplates.size(), importedTablesCount);
            //
            checkInterrupted();
            if (exportConfig.importAllTables()) {
                for (String dbName : exportConfig.getDbNames()) {
                    if (dynamicTable.getTargetTable().contains("/" + dbName + "/")) {
                        createDynamicTable(dynamicTable, Collections.singletonList(dbName));
                        break;
                    }
                }
            } else {
                createDynamicTable(dynamicTable, state.dbNames);
            }
            //
            importedTablesCount++;
        }

        for (TableExportTemplate dynamicIndex : tableIndexExportTemplates) {
            checkInterrupted();
            createDynamicIndex(dynamicIndex);
        }

        // Помечаем все учавствовавшие в импорте базы как проимпортированные
        for (String dbName : exportConfig.getDbNames()) {
            state.markImported(dbName);
        }
        checkInterrupted();
        state.commit(ytSupport);
    }

    /**
     * Проверяет необходимость и возможность импорта базы dbName
     */
    private void checkDatabase(InitialImportState state, String dbName) throws InterruptedException, SQLException {
        if (!dbName.startsWith(exportConfig.getBaseSchemaName() + ":")
                && !(exportConfig.importAllTables() && dbName.equals(exportConfig.getPpcDictSchemaName()))) {
            throw new IllegalArgumentException("Cannot import from " + dbName);
        }
        state.dbNames.add(dbName);

        MySQLBinlogState binlogState = state.getBinlogState(dbName);
        if (binlogState == null) {
            checkInterrupted();
            // Делаем снепшот базы dbName
            try (Connection conn = connectionsCache.getConnection(dbName)) {
                logger.info("Making a snapshot of {}...", dbName);
                binlogState = MySQLBinlogState.snapshot(conn);
                state.setBinlogState(dbName, binlogState);
            }
        }

        if (state.isImported(dbName)) {
            ++state.importedCount;
        } else {
            ++state.unimportedCount;
        }
    }

    private void checkInterrupted() throws InterruptedException {
        if (Thread.interrupted()) {
            throw new InterruptedException();
        }
    }

    private YPath staticImportTablePath(String tableName) {
        if (exportConfig.importAllTables()) {
            String dbNames = YtSyncCommonUtil.getAllTablesImportInitialSubDir(exportConfig.getDbNames());
            return YPath.simple(exportConfig.rootPath() + "/" + dbNames + exportConfig.getTablesPrefix() + tableName);
        } else {
            return YPath.simple(exportConfig.rootPath() + exportConfig.getTablesPrefix() + tableName);
        }
    }

    private void createDynamicTable(TableExportTemplate dynamicTable, List<String> dbNames)
            throws InterruptedException {
        YPath dynamicTargetPath = YPath.simple(dynamicTable.getTargetTable());
        if (!yt.cypress().exists(dynamicTargetPath)) {
            // Динамической таблицы не существует, делаем импорт статических таблиц
            List<YPath> inputTables = new ArrayList<>();
            List<ShardedTableImporter> importersFinished = new ArrayList<>();
            for (String staticSourceName : dynamicTable.getSourceTables()) {
                checkInterrupted();
                YPath staticTablePath = staticImportTablePath(staticSourceName);
                // Если есть отсортированная версия, значит пропускаем этот шаг
                if (!staticToDynamicConverter.sortedTableExists(dynamicTargetPath, null)) {
                    ShardedTableImporter importer = createStaticTable(staticTablePath, dynamicTargetPath,
                            staticSourceName, dbNames);
                    importersFinished.add(importer);
                }
                inputTables.add(staticTablePath);
            }

            // Делаем трансформацию статических таблиц в динамическую
            logger.info("Creating dynamic table {}", dynamicTargetPath);
            staticToDynamicConverter
                    .convert(inputTables, dynamicTargetPath, dynamicTable.getSchema(), dynamicTable.getPivotKeys(),
                            true);

            // Удаляем таблицы состояний, они больше не нужны
            for (ShardedTableImporter importer : importersFinished) {
                importer.deleteStateTable();
            }
        }
    }

    /**
     * Выполняет наливку статической несортированной таблицы.
     * Если наливка успешно завершается, то возвращается импортёр, который выполнял импорт.
     * Это позволит затем, после сортировки, удалить таблицу состояния импорта, которая уже будет не нужна.
     */
    private ShardedTableImporter createStaticTable(YPath staticTablePath, YPath dynamicTargetPath, String tableName,
                                                   List<String> dbNames)
            throws InterruptedException {
        TableExportTask template = tableNameAndPathToTemplate.get(Pair.of(tableName, dynamicTargetPath.toString()));
        if (template == null) {
            throw new IllegalStateException("Missing template for static table " + tableName);
        }
        ShardedTableImporter importer = new ShardedTableImporter(
                yt, compressorFactory, connectionsCache,
                exportConfig.getBaseSchemaName(),
                template,
                exportConfig.chunkSize(),
                exportConfig.chunksLimit(),
                staticTablePath,
                commonStaticAttributes(),
                exportConfig.threadsNum(dbNames),
                exportConfig.flushIntervalMinutes()
        );
        currentImporter = importer;
        importer.run(dbNames);
        currentImporter = null;
        return importer;
    }

    private void createDynamicIndex(TableExportTemplate dynamicIndex) {
        List<YPath> sourceTablePaths = mapList(dynamicIndex.getSourceTables(), YPath::simple);
        YPath indexTargetPath = YPath.simple(dynamicIndex.getTargetTable());
        if (!yt.cypress().exists(indexTargetPath)) {
            // Индекса не существует, создаём его
            for (YPath sourceTablePath : sourceTablePaths) {
                if (!yt.cypress().exists(sourceTablePath)) {
                    throw new IllegalStateException("Source table " + sourceTablePath + " does not exist");
                }
            }

            logger.info("Creating dynamic index {}", indexTargetPath);
            staticToDynamicConverter
                    .convert(sourceTablePaths, indexTargetPath, dynamicIndex.getSchema(),
                            dynamicIndex.getPivotKeys(), false);
        }
    }

    @PreDestroy
    public void stop() {
        logger.info("Stopping table exporter {}", this);
        ShardedTableImporter currentImporter = this.currentImporter;
        if (currentImporter != null) {
            currentImporter.stop();
        }
    }

    private MapF<String, YTreeNode> commonStaticAttributes() {
        return Cf.map(
                "optimize_for", YTree.stringNode(exportConfig.getOptimizeFor().getText()),
                "primary_medium", YTree.stringNode(exportConfig.ytTablesMedium()));
    }

    private static class InitialImportState {
        private final SyncStatesTable syncStatesTable;
        private final List<String> dbNames = new ArrayList<>();
        private boolean saveNeeded;
        private int unimportedCount;
        private int importedCount;

        private InitialImportState(SyncStatesTable syncStatesTable) {
            this.syncStatesTable = syncStatesTable;
        }

        private void prepareTable() {
            syncStatesTable.prepareTable();
        }

        private MySQLBinlogState getBinlogState(String dbName) {
            return syncStatesTable.getBinlogState(dbName);
        }

        private void setBinlogState(String dbName, MySQLBinlogState syncState) {
            syncStatesTable.setBinlogState(dbName, syncState);
            saveNeeded = true;
        }

        private boolean isImported(String dbName) {
            return syncStatesTable.isImported(dbName);
        }

        private void markImported(String dbName) {
            syncStatesTable.setImported(dbName, true);
            saveNeeded = true;
        }

        private void commit(YtSupport ytSupport) {
            if (saveNeeded) {
                ytSupport.runTransaction(syncStatesTable::apply).join(); // IGNORE-BAD-JOIN DIRECT-149116
                syncStatesTable.committed();
                saveNeeded = false;
            }
        }
    }

    private static class InitialImportMetrics {
        final int totalTablesCount;
        final int importedTablesCount;

        InitialImportMetrics(int totalTablesCount, int importedTablesCount) {
            this.totalTablesCount = totalTablesCount;
            this.importedTablesCount = importedTablesCount;
        }
    }

    public void addInitialImportMetrics(MetricRegistry metricRegistry) {
        metricRegistry.lazyGaugeInt64("initial_import.tablesTotalCnt", () -> {
            InitialImportMetrics metricsState = this.metricsState;
            return metricsState != null ? metricsState.totalTablesCount : 0;
        });
        metricRegistry.lazyGaugeInt64("initial_import.tablesImportedCnt", () -> {
            InitialImportMetrics metricsState = this.metricsState;
            return metricsState != null ? metricsState.importedTablesCount : 0;
        });
    }
}
