package ru.yandex.direct.mysql.ytsync.export.task;

import java.util.Objects;

import ru.yandex.direct.mysql.ytsync.export.util.valueprocessing.ExportFlatRowCreator;
import ru.yandex.yt.ytclient.tables.ColumnSchema;
import ru.yandex.yt.ytclient.tables.TableSchema;

/**
 * Шаблонные параметрами для экспорта одной таблицы
 */
public class TableExportTask {
    private final String sqlTemplate;
    private final String tableName;
    private final String idColumn;
    private final TableSchema schema;
    private final ExportFlatRowCreator exportFlatRowCreator;
    private final TableSchema writeSchema;
    private final TableSchema staticSchema;

    public TableExportTask(String sqlTemplate, String tableName, String idColumn, TableSchema schema,
                           ExportFlatRowCreator exportFlatRowCreator) {
        this.sqlTemplate = Objects.requireNonNull(sqlTemplate);
        this.tableName = Objects.requireNonNull(tableName);
        this.idColumn = Objects.requireNonNull(idColumn);
        this.schema = Objects.requireNonNull(schema);
        this.exportFlatRowCreator = Objects.requireNonNull(exportFlatRowCreator);
        this.writeSchema = this.schema.toWrite();
        this.staticSchema = makeStaticSchema(this.schema);
    }

    public String getSqlTemplate() {
        return sqlTemplate;
    }

    public String getTableName() {
        return tableName;
    }

    public String getIdColumn() {
        return idColumn;
    }

    /**
     * Возвращает идеальную схему таблицы
     */
    public TableSchema getSchema() {
        return schema;
    }

    /**
     * Возвращает схему без expression колонок, для записи данных в таблицу
     */
    public TableSchema getWriteSchema() {
        return writeSchema;
    }

    /**
     * Возвращает схему для создания статической таблицы, без сортировки
     */
    public TableSchema getStaticSchema() {
        return staticSchema;
    }

    public ExportFlatRowCreator getExportFlatRowCreator() {
        return exportFlatRowCreator;
    }

    private static TableSchema makeStaticSchema(TableSchema schema) {
        TableSchema.Builder staticSchemaBuilder = new TableSchema.Builder();
        staticSchemaBuilder.setUniqueKeys(false);
        for (ColumnSchema column : schema.getColumns()) {
            staticSchemaBuilder.add(new ColumnSchema(
                    column.getName(),
                    column.getType(),
                    null,
                    column.getLock(),
                    column.getExpression(),
                    column.getAggregate(),
                    column.getGroup(),
                    false));
        }
        return staticSchemaBuilder.build();
    }
}
