package ru.yandex.direct.mysql.ytsync.export.task;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Iterables;

import ru.yandex.direct.mysql.ytsync.common.keys.PivotKeys;
import ru.yandex.yt.ytclient.tables.TableSchema;

@ParametersAreNonnullByDefault
public class TableExportTemplate {
    private final List<String> sourceTables;
    private final String targetTable;
    private final TableSchema schema;
    private final PivotKeys pivotKeys;

    public TableExportTemplate(String sourceTable, String targetTable, TableSchema schema) {
        this(Collections.singletonList(sourceTable), targetTable, schema, null);
    }

    public TableExportTemplate(List<String> sourceTables, String targetTable,
                               TableSchema schema, @Nullable PivotKeys pivotKeys) {
        this.targetTable = targetTable;
        this.sourceTables = sourceTables;
        this.schema = schema;
        this.pivotKeys = pivotKeys;
    }

    public String getTargetTable() {
        return targetTable;
    }

    public List<String> getSourceTables() {
        return sourceTables;
    }

    public TableSchema getSchema() {
        return schema;
    }

    public PivotKeys getPivotKeys() {
        return pivotKeys;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        TableExportTemplate that = (TableExportTemplate) o;
        return Objects.equals(sourceTables, that.sourceTables) &&
                Objects.equals(targetTable, that.targetTable) &&
                Objects.equals(schema, that.schema) &&
                Objects.equals(pivotKeys, that.pivotKeys);
    }

    @Override
    public int hashCode() {
        return Objects.hash(sourceTables, targetTable, schema, pivotKeys);
    }

    public static TableExportTemplate merge(Collection<TableExportTemplate> tableExportTemplates) {
        if (tableExportTemplates.isEmpty()) {
            return null;
        } else if (tableExportTemplates.size() == 1) {
            return Iterables.getOnlyElement(tableExportTemplates);
        }

        List<TableExportTemplate> templates = new ArrayList<>(tableExportTemplates);

        TableExportTemplate first = templates.get(0);
        List<String> sources = new ArrayList<>(first.sourceTables);

        for (int i = 1; i < tableExportTemplates.size(); i++) {
            TableExportTemplate item = templates.get(i);
            if (!Objects.equals(first.targetTable, item.targetTable)
                    || !Objects.equals(first.schema, item.schema)
                    || !Objects.equals(first.pivotKeys, item.pivotKeys)
            ) {
                throw new IllegalArgumentException("Source tables " + first.getSourceTables().get(0)
                        + " and " + item.getSourceTables().get(0) + " cannot be merged");
            }
            sources.addAll(item.sourceTables);
        }

        return new TableExportTemplate(sources, first.targetTable, first.schema, first.pivotKeys);
    }
}
