package ru.yandex.direct.mysql.ytsync.export.util;

import java.util.Iterator;
import java.util.concurrent.atomic.AtomicLong;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.mysql.ytsync.common.util.ParallelRunner;
import ru.yandex.direct.mysql.ytsync.export.util.queue.ShardedQueue;
import ru.yandex.direct.utils.InterruptedRuntimeException;

/**
 * Вспомогательный поток, который следит за скоростью потребления строк и чанков
 */
class ProcessingSpeedCounter {
    private static final Logger logger = LoggerFactory.getLogger(ProcessingSpeedCounter.class);
    private static final double NANOSECONDS_PER_SECOND = 1_000_000_000.0;

    private final ShardedQueue<IdRange> rangesQueue;
    private final AtomicLong rowsCount = new AtomicLong();
    private final Thread thread;
    private volatile ParallelRunner<ShardedSqlLoader> parallelRunner;

    ProcessingSpeedCounter(ShardedQueue<IdRange> rangesQueue) {
        this.rangesQueue = rangesQueue;
        //
        thread = new Thread(this::run);
        thread.setDaemon(true);
        thread.start();
    }

    void setParallelRunner(ParallelRunner<ShardedSqlLoader> parallelRunner) {
        this.parallelRunner = parallelRunner;
    }

    private void run() {
        long startTimestamp = System.nanoTime();
        long chunkStartTimestamp = startTimestamp;
        long chunkStartCount = 0;
        boolean done = false;
        while (!done) {
            try {
                Thread.sleep(1000);
            } catch (InterruptedException e) {
                done = true;
            }
            long now = System.nanoTime();
            if (now > chunkStartTimestamp) {
                long totalCount = rowsCount.get();
                double totalEla = (now - startTimestamp) / NANOSECONDS_PER_SECOND;
                double totalRps = totalCount / totalEla;
                double chunkEla = (now - chunkStartTimestamp) / NANOSECONDS_PER_SECOND;
                double chunkRps = (totalCount - chunkStartCount) / chunkEla;
                ParallelRunner<ShardedSqlLoader> parallelRunner = this.parallelRunner;
                int activeThreadsCount = 0;
                if (parallelRunner != null) {
                    activeThreadsCount = parallelRunner.getAliveThreadsCount();
                }
                logger.info(String.format(
                        "Processed %d rows, %.2f rows/second, avg. %.2f rows/second. %d chunks remaining. Active " +
                                "threads %d",
                        totalCount,
                        chunkRps,
                        totalRps,
                        rangesQueue.getChunksCount(),
                        activeThreadsCount));
                chunkStartTimestamp = now;
                chunkStartCount = totalCount;
            }
        }
    }

    void stop() throws InterruptedException {
        thread.interrupt();
        thread.join(); // IS-NOT-COMPLETABLE-FUTURE-JOIN
    }

    <T> Iterator<T> spyWithInterrupts(Iterator<T> source) {
        return new Iterator<T>() {
            @Override
            public boolean hasNext() {
                return source.hasNext();
            }

            @Override
            public T next() {
                if (Thread.interrupted()) {
                    Thread.currentThread().interrupt();
                    throw new InterruptedRuntimeException();
                }
                T value = source.next();
                rowsCount.incrementAndGet();
                return value;
            }
        };
    }
}
