package ru.yandex.direct.mysql.ytsync.export.util;

import java.sql.Connection;
import java.sql.SQLException;
import java.util.Iterator;
import java.util.NoSuchElementException;
import java.util.Objects;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.mysql.ytsync.export.components.ConnectionsCache;
import ru.yandex.direct.mysql.ytsync.export.util.queue.ShardedQueue;
import ru.yandex.direct.mysql.ytsync.export.util.queue.ShardedQueueValue;
import ru.yandex.direct.mysql.ytsync.export.util.valueprocessing.ExportFlatRowCreator;

/**
 * Шардированный итератор загрузчиков по диапазонам
 * <p>
 * Создаёт YtSqlLoader для очередного диапазона на каждом вызове next.
 * <p>
 * Автоматически освобождает ресурсы на вызовах hasNext.
 * <p>
 * Мб переименовать в ???
 * Мб сделать Closeable ?
 */
public class ShardedSqlLoader implements Iterator<YtSqlLoader> {
    private static final Logger logger = LoggerFactory.getLogger(ShardedSqlLoader.class);

    private final ConnectionsCache connectionsCache;
    private final ShardedQueue<IdRange> shardedQueue;
    private final String sqlTemplate;
    private final ExportFlatRowCreator exportFlatRowCreator;

    private ShardedQueueValue<IdRange> currentItem;
    private boolean fetched;
    private String currentDbName;
    private Connection currentConnection;
    private YtSqlLoader currentSqlLoader;

    private volatile boolean paused = false;

    ShardedSqlLoader(
            ConnectionsCache connectionsCache,
            ShardedQueue<IdRange> shardedQueue,
            String sqlTemplate,
            ExportFlatRowCreator exportFlatRowCreator
    ) {
        this.connectionsCache = Objects.requireNonNull(connectionsCache);
        this.shardedQueue = Objects.requireNonNull(shardedQueue);
        this.sqlTemplate = Objects.requireNonNull(sqlTemplate);
        this.exportFlatRowCreator = Objects.requireNonNull(exportFlatRowCreator);
    }

    private void releaseSqlLoader() throws SQLException {
        if (currentSqlLoader != null) {
            try {
                currentSqlLoader.close();
            } finally {
                currentSqlLoader = null;
            }
        }
    }

    private void releaseConnection() throws SQLException {
        if (currentConnection != null) {
            try {
                connectionsCache.releaseConnection(currentDbName, currentConnection);
            } finally {
                currentConnection = null;
                currentDbName = null;
            }
        }
    }

    void releaseResources() {
        try {
            try {
                releaseSqlLoader();
            } finally {
                releaseConnection();
            }
        } catch (SQLException e) {
            logger.warn("Failed to release resources", e);
        }
    }

    private void prefetch() {
        if (!fetched) {
            // Освобождаем ресурсы предыдущей итерации
            releaseResources();
            // Освобождаем предыдущий диапазон
            if (currentItem != null) {
                currentItem.release();
                currentItem = null;
            }
            // Берём новый диапазон
            if (!paused) {
                currentItem = shardedQueue.poll();
                fetched = true;
            }
        }
    }

    @Override
    public boolean hasNext() {
        prefetch();
        return currentItem != null;
    }

    @Override
    public YtSqlLoader next() {
        prefetch();
        if (currentItem == null) {
            throw new NoSuchElementException();
        }
        IdRange range = currentItem.getValue();
        String dbName = currentItem.getDbName();
        String sql = sqlTemplate
                .replace("{MIN_ID}", "'" + range.getMinId() + "'")
                .replace("{MAX_ID}", "'" + range.getMaxId() + "'");
        logger.info("Creating {} loader with SQL: {}", dbName, sql);
        try {
            currentDbName = dbName;
            currentConnection = connectionsCache.getConnection(dbName);
        } catch (SQLException e) {
            // FIXME(snaury): придумать другое исключение
            throw new RuntimeException(e);
        }
        fetched = false;
        try {
            currentSqlLoader = new YtSqlLoader(currentConnection, sql, dbName, exportFlatRowCreator);
            return currentSqlLoader;
        } catch (SQLException e) {
            releaseResources();
            // FIXME(snaury): придумать другое исключение
            throw new RuntimeException(e);
        }
    }

    /**
     * Приостанавливает процесс получения новых чанков из очереди shardedQueue.
     * В состоянии paused итератор отдаст только один чанк, тот, который он успел взять из очереди.
     * Для продолжения нужно позвать метод {@link #resume()}.
     */
    void pause() {
        paused = true;
    }

    /**
     * Возобновляет процесс получения новых чанков из очереди.
     */
    void resume() {
        paused = false;
    }
}
