package ru.yandex.direct.mysql.ytsync.export.util.queue;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.Nullable;

/**
 * Индексируемая очередь с приоритетами.
 * Значения хранятся в binary min-heap, а рядом лежит соответствие ключа индексу элемента в куче.
 * Поддерживаются операции "получить элемент по ключу", "получить минимальный элемент",
 * "изменить элемент" (можно выполнить как decreaseKey, так и increaseKey),
 * "удалить по ключу".
 *
 * @param <TKey>   Ключ. Должен быть применим в качестве ключа HashMap
 * @param <TValue> Значение. Должно быть Comparable
 */
public class IndexedPriorityQueue<TKey, TValue extends Comparable<TValue>> {
    private Map<TKey, Integer> indexes = new HashMap<>();
    private List<KVPair> heap = new ArrayList<>();

    public class KVPair implements Comparable<KVPair> {
        public final TKey key;
        public final TValue value;

        KVPair(TKey key, TValue value) {
            this.key = key;
            this.value = value;
        }

        @Override
        public int compareTo(KVPair o) {
            return this.value.compareTo(o.value);
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            KVPair kvPair = (KVPair) o;
            return key.equals(kvPair.key) &&
                    value.equals(kvPair.value);
        }

        @Override
        public int hashCode() {
            return Objects.hash(key, value);
        }
    }

    private void siftUp(int index) {
        while (index > 0 && heap.get(index).compareTo(heap.get((index - 1) >> 1)) < 0) {
            swap(index, (index - 1) >> 1);
            index = (index - 1) >> 1;
        }
    }

    private void siftDown(int index) {
        while ((index << 1) + 1 < heap.size()) {
            int child = (index << 1) + 1;  // Сначала левый сын
            if (child + 1 < heap.size() && heap.get(child + 1).compareTo(heap.get(child)) < 0) {
                // Если правый меньше -- берём его
                child++;
            }
            if (heap.get(index).compareTo(heap.get(child)) <= 0) {
                break;
            }
            swap(index, child);
            index = child;
        }
    }

    private void swap(int first, int second) {
        KVPair tmp = heap.get(first);
        heap.set(first, heap.get(second));
        heap.set(second, tmp);
        indexes.put(heap.get(first).key, first);
        indexes.put(heap.get(second).key, second);
    }

    public void put(TKey key, TValue value) {
        int index;
        if (!indexes.containsKey(key)) {
            index = heap.size();
            heap.add(new KVPair(key, value));
            indexes.put(key, index);
        } else {
            index = indexes.get(key);
            heap.set(indexes.get(key), new KVPair(key, value));
        }
        siftUp(index);
        siftDown(indexes.get(key));
    }

    public void change(TKey key, Function<TValue, TValue> mapper) {
        if (!containsKey(key)) {
            throw new NoSuchElementException();
        }
        TValue oldValue = heap.get(indexes.get(key)).value;
        TValue newValue = mapper.apply(oldValue);
        put(key, newValue);
    }

    @Nullable
    public KVPair extractMin() {
        if (heap.isEmpty()) {
            return null;
        }
        KVPair result = heap.get(0);
        swap(0, heap.size() - 1);
        indexes.remove(result.key);
        heap.remove(heap.size() - 1);
        siftDown(0);
        return result;
    }

    @Nullable
    public KVPair peekMin() {
        return heap.isEmpty() ? null : heap.get(0);
    }

    public boolean isEmpty() {
        return heap.isEmpty();
    }

    /**
     * Возвращает элемент по ключу.
     */
    @Nullable
    public TValue get(TKey key) {
        return indexes.containsKey(key) ? heap.get(indexes.get(key)).value : null;
    }

    public boolean containsKey(TKey key) {
        return indexes.containsKey(key);
    }

    /**
     * Удаляет элемент по ключу
     */
    public void delete(TKey key) {
        if (indexes.containsKey(key)) {
            Integer index = indexes.get(key);
            swap(index, heap.size() - 1);
            indexes.remove(key);
            heap.remove(heap.size() - 1);
            if (index < heap.size()) {
                TKey swappedKey = heap.get(index).key;
                siftUp(index);
                siftDown(indexes.get(swappedKey));
            }
        }
    }

    public Set<TKey> keySet() {
        return indexes.keySet();
    }

    public int size() {
        return heap.size();
    }

    public void clear() {
        indexes.clear();
        heap.clear();
    }
}
