package ru.yandex.direct.mysql.ytsync.export.util.queue;

import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Deque;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nonnull;

/**
 * Очередь, распределяющая нагрузку по шардам
 * <p>
 * Инкапсулирует несколько очередей по шардам, пытается отдавать значения на одновременную обработку
 * по возможности из разных шардов. Основное использование - распараллеливание обработки данных, когда
 * одновременная обработка на разных шардах предпочтительнее, однако кол-во работы по шардам распределено
 * неравномерно. При этом в конце обработки будет распараллеливаться остаток из самых крупных шардов.
 */
public class ShardedQueue<T> {
    // shard -> ShardData
    private final IndexedPriorityQueue<String, ShardData<T>> priorityQueue = new IndexedPriorityQueue<>();

    public synchronized ShardedQueueValue<T> poll() {
        if (priorityQueue.isEmpty()) {
            // У нас нет ни одного шарда
            return null;
        }
        ShardData<T> data = priorityQueue.peekMin().value;
        T value = data.queue.poll();
        if (value == null) {
            // Все пустые шарды лежат в конце
            return null;
        }
        priorityQueue.change(data.shard, shardData -> {
            shardData.taken++;
            return shardData;
        });
        return new ShardedQueueValue<>(data.shard, value, this::releaseShard);
    }

    public synchronized boolean isEmpty() {
        return priorityQueue.isEmpty() || priorityQueue.peekMin().value.queue.isEmpty();
    }

    /**
     * Возвращает количество чанков, которое осталось обработать.
     */
    public synchronized long getChunksCount() {
        Set<String> shards = priorityQueue.keySet();
        return shards.stream().mapToInt(shard -> priorityQueue.get(shard).queue.size()).sum();
    }

    /**
     * Добавляет элементы из коллекции c в шард shard
     */
    public synchronized void add(String dbName, Collection<T> c) {
        ShardData<T> data = priorityQueue.get(dbName);
        if (data == null) {
            data = new ShardData<>(dbName);
            data.queue.addAll(c);
            priorityQueue.put(dbName, data);
        } else {
            priorityQueue.change(dbName, shardData -> {
                shardData.queue.addAll(c);
                return shardData;
            });
        }
    }

    /**
     * Возвращает копию текущего состояния очереди, позволяя сохранить её для дальнейшего восстановления.
     */
    public synchronized Map<String, List<T>> copyToMap() {
        Map<String, List<T>> map = new HashMap<>();
        for (String key : priorityQueue.keySet()) {
            map.put(key, new ArrayList<T>(priorityQueue.get(key).queue));
        }
        return map;
    }

    /**
     * Восстанавливает состояние из переданной мапы, целиком перезатирая текущее.
     */
    public synchronized void loadFromMap(Map<String, List<T>> map) {
        priorityQueue.clear();
        for (Map.Entry<String, List<T>> entry : map.entrySet()) {
            ShardData<T> shardData = new ShardData<>(entry.getKey());
            shardData.queue.addAll(entry.getValue());
            priorityQueue.put(entry.getKey(), shardData);
        }
    }

    private static class ShardData<T> implements Comparable<ShardData<T>> {
        private final String shard;
        private int taken;
        private final Deque<T> queue = new ArrayDeque<>();

        private ShardData(String shard) {
            this.shard = shard;
        }

        @Override
        public int compareTo(@Nonnull ShardData<T> o) {
            int result = Boolean.compare(queue.isEmpty(), o.queue.isEmpty());
            if (result == 0) {
                result = Integer.compare(taken, o.taken);
                if (result == 0) {
                    result = CharSequence.compare(shard, o.shard);
                }
            }
            return result;
        }
    }

    private synchronized void releaseShard(String shard) {
        priorityQueue.change(shard, shardData -> {
            shardData.taken--;
            return shardData;
        });
    }
}
