package ru.yandex.direct.mysql.ytsync.export.util.valueprocessing;

import java.math.BigDecimal;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.function.Function;

import ru.yandex.direct.utils.DateTimeUtils;
import ru.yandex.misc.lang.number.UnsignedLong;

public interface ResultSetValueExtractor<T> extends Function<ResultSetPair, T> {
    static ResultSetValueExtractor<Long> longExtractor() {
        return rsp -> {
            try {
                long value = rsp.getResultSet().getLong(rsp.getIndex());
                return rsp.getResultSet().wasNull() ? null : value;
            } catch (SQLException e) {
                throw new RuntimeException();
            }
        };
    }

    static ResultSetValueExtractor<String> stringExtractor() {
        return rsp -> {
            try {
                return rsp.getResultSet().getString(rsp.getIndex());
            } catch (SQLException e) {
                throw new RuntimeException();
            }
        };
    }

    static ResultSetValueExtractor<BigDecimal> doubleExtractor() {
        return rsp -> {
            try {
                return rsp.getResultSet().getBigDecimal(rsp.getIndex());
            } catch (SQLException e) {
                throw new RuntimeException();
            }
        };
    }

    static ResultSetValueExtractor<LocalDate> localDateExtractor() {
        return rsp -> {
            Timestamp timestamp = getTimestamp(rsp);
            return timestamp != null
                    ? timestamp.toInstant().atZone(DateTimeUtils.MSK).toLocalDateTime().toLocalDate()
                    : null;
        };
    }

    static ResultSetValueExtractor<LocalDateTime> localDateTimeExtractor() {
        return rsp -> {
            Timestamp timestamp = getTimestamp(rsp);
            return timestamp != null
                    ? timestamp.toInstant().atZone(DateTimeUtils.MSK).toLocalDateTime()
                    : null;
        };
    }

    private static Timestamp getTimestamp(ResultSetPair resultSetPair) {
        try {
            return resultSetPair.getResultSet().getTimestamp(resultSetPair.getIndex());
        } catch (SQLException e) {
            throw new RuntimeException();
        }
    }

    static ResultSetValueExtractor<UnsignedLong> unsignedLongExtractor() {
        return rsp -> {
            try {
                String v = rsp.getResultSet().getString(rsp.getIndex());
                if (v != null) {
                    return UnsignedLong.valueOf(Long.parseUnsignedLong(v, 10));
                }
                return null;
            } catch (SQLException e) {
                throw new RuntimeException();
            }
        };
    }
}
