package ru.yandex.direct.mysql.ytsync.common.compatibility;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ScheduledExecutorService;

import ru.yandex.direct.mysql.ytsync.common.keys.PivotKeys;
import ru.yandex.direct.mysql.ytsync.common.row.FlatRow;
import ru.yandex.direct.mysql.ytsync.common.row.FlatRowView;
import ru.yandex.direct.ytwrapper.YtUtils;
import ru.yandex.direct.ytwrapper.model.attributes.OptimizeForAttr;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTree;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTreeBuilder;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;
import ru.yandex.yt.ytclient.tables.TableSchema;
import ru.yandex.yt.ytclient.wire.UnversionedRowset;

import static com.google.common.base.Preconditions.checkState;

/**
 * Абстракция над базовыми операциями yt
 */
public interface BasicYtSupport {
    /**
     * Даёт доступ к executor'у для таймаутов и прочего
     */
    ScheduledExecutorService executor();

    /**
     * Возвращает true, если нода по пути path существует
     */
    CompletableFuture<Boolean> exists(String path);

    /**
     * Возвращает содержимое ноды по пути path
     */
    CompletableFuture<YTreeNode> getNode(String path);

    CompletableFuture<YTreeNode> getNode(String path, Set<String> attributes);

    /**
     * Устанавливает значение ноды в указанное значение
     */
    CompletableFuture<Void> setNode(String path, YTreeNode value);

    /**
     * Берёт эксклюзивный лок на указанную ноду
     */
    CompletableFuture<YtLock> lockNodeExclusive(String path, String title, boolean waitable);

    /**
     * Удаляет объект по пути path
     */
    CompletableFuture<Void> remove(String path);

    /**
     * Создаёт таблицу с указанными атрибутами
     */
    CompletableFuture<Void> createTable(String path, Map<String, YTreeNode> attributes);

    /**
     * Создаёт динамическую таблицу с указанной схемой, оптимизацией и атрибутами
     */
    default CompletableFuture<Void> createDynamicTable(String path, TableSchema schema, OptimizeForAttr optimizeFor,
                                                       Map<String, YTreeNode> customAttributes) {
        YTreeBuilder builder = YTree.builder();
        builder.beginMap()
                .key("optimize_for").value(optimizeFor.getText())
                .key("dynamic").value(true)
                .key(YtUtils.SCHEMA_ATTR).value(schema.toYTree())
                .endMap();
        Map<String, YTreeNode> attributes = builder.build().asMap();
        for (Map.Entry<String, YTreeNode> entry : customAttributes.entrySet()) {
            checkState(!attributes.containsKey(entry.getKey()));
            attributes.put(entry.getKey(), entry.getValue());
        }
        return createTable(path, attributes);
    }

    /**
     * Перешардирует таблицу используя указанные ключи
     */
    CompletableFuture<Void> reshardTable(String path, PivotKeys pivotKeys);

    /**
     * Монтирует таблицу по пути path
     */
    CompletableFuture<Void> mountTable(String path);

    /**
     * Размораживает таблицу по пути path
     */
    CompletableFuture<Void> unfreezeTable(String path);

    /**
     * Размонтирует таблицу по пути path
     */
    CompletableFuture<Void> unmountTable(String path);

    /**
     * Выполняет произвольный select запрос и возвращает результат в схеме resultSchema
     */
    CompletableFuture<List<FlatRow>> selectRows(String query, TableSchema resultSchema);

    CompletableFuture<UnversionedRowset> selectRows(String query);

    /**
     * Возвращает null-транзакцию, в которой все операции выполняются сразу и не атомарно
     */
    CompletableFuture<? extends BasicTransaction> nullTransaction();

    /**
     * Создаёт транзакцию для работы с данными
     */
    CompletableFuture<? extends BasicTransaction> startTransaction();

    /**
     * Абстракция над базовыми операциями в транзакциях
     */
    interface BasicTransaction {
        /**
         * Даёт доступ к владельцу транзакции
         */
        BasicYtSupport support();

        /**
         * Даёт доступ к executor'у для таймаутов и прочего
         */
        default ScheduledExecutorService executor() {
            return support().executor();
        }

        /**
         * Возвращает true, если транзакции атомарные (false означает фактическое отсутствие транзакций)
         */
        boolean isAtomic();

        /**
         * Пингует транзакцию
         */
        CompletableFuture<Void> ping();

        /**
         * Отменяет транзакцию
         */
        CompletableFuture<Void> abort();

        /**
         * Коммитит транзакцию
         */
        CompletableFuture<Void> commit();

        /**
         * Вставляет в таблицу path строки со схемой schema
         */
        CompletableFuture<Void> insertRows(String path, TableSchema schema, List<? extends FlatRowView> rows);

        /**
         * Обновляет в таблице path строки со схемой schema
         */
        CompletableFuture<Void> updateRows(String path, TableSchema schema, List<? extends FlatRowView> rows);

        /**
         * Удаляет в таблице path ключи со схемой schema
         */
        CompletableFuture<Void> deleteRows(String path, TableSchema schema, List<? extends FlatRowView> keys);

        /**
         * Меняет строки в таблице path используя данные со схемой schema
         */
        CompletableFuture<Void> modifyRows(String path, TableSchema schema, List<? extends FlatRowView> insertedRows,
                                           List<? extends FlatRowView> updatedRows, List<? extends FlatRowView> deletedKeys);

        /**
         * Находит строки в таблице path используя ключи keys со схемой keySchema
         * <p>
         * Возвращаются строки со схемой resultSchema
         */
        CompletableFuture<List<FlatRow>> lookupRows(String path, TableSchema keySchema,
                                                    List<? extends FlatRowView> keys, TableSchema resultSchema);
    }
}
