package ru.yandex.direct.mysql.ytsync.common.compatibility;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Consumer;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.IterableF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.direct.mysql.ytsync.common.keys.PivotKeys;
import ru.yandex.direct.mysql.ytsync.common.row.FlatRow;
import ru.yandex.direct.mysql.ytsync.common.row.FlatRowView;
import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.inside.yt.kosher.common.GUID;
import ru.yandex.inside.yt.kosher.cypress.CypressNodeType;
import ru.yandex.inside.yt.kosher.cypress.LockMode;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTree;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;
import ru.yandex.yt.ytclient.tables.TableSchema;
import ru.yandex.yt.ytclient.wire.UnversionedRowset;

import static ru.yandex.direct.mysql.ytsync.common.util.YtSyncCommonUtil.YTSYNC_LOGGER;

/**
 * Реализация BasicYtSupport используя кошерный inside-yt клиент
 */
public class BasicYtSupportViaKosher implements BasicYtSupport {
    private static final Logger logger = LoggerFactory.getLogger(BasicYtSupportViaKosher.class);
    private static final int TRANSACTION_TIMEOUT_SECONDS = 30;

    private final Yt yt;
    private final ScheduledExecutorService executor;
    private final Executor syncExecutor;
    private final Executor cachedExecutor;
    private final KosherCompressorFactory compressorFactory;

    public BasicYtSupportViaKosher(Yt yt, ScheduledExecutorService executor, Executor syncExecutor,
                                   Executor cachedExecutor, KosherCompressorFactory compressorFactory) {
        this.yt = yt;
        this.executor = executor;
        this.syncExecutor = syncExecutor;
        this.cachedExecutor = cachedExecutor;
        this.compressorFactory = compressorFactory;
    }

    @Override
    public ScheduledExecutorService executor() {
        return executor;
    }

    @Override
    public CompletableFuture<Boolean> exists(String path) {
        return CompletableFuture.supplyAsync(() -> yt.cypress().exists(YPath.simple(path)), syncExecutor);
    }

    @Override
    public CompletableFuture<YTreeNode> getNode(String path) {
        return CompletableFuture
                .supplyAsync(() -> yt.cypress().get(YPath.simple(path)),
                        syncExecutor);
    }

    @Override
    public CompletableFuture<YTreeNode> getNode(String path, Set<String> attributes) {
        return CompletableFuture.supplyAsync(
                () -> yt.cypress().get(YPath.simple(path), Cf.wrap(attributes)),
                syncExecutor
        );
    }

    @Override
    public CompletableFuture<Void> setNode(String path, YTreeNode value) {
        return CompletableFuture
                .runAsync(() -> yt.cypress().set(YPath.simple(path), value),
                        syncExecutor);
    }

    @Override
    public CompletableFuture<YtLock> lockNodeExclusive(String path, String title, boolean waitable) {
        final CompletableFuture<YtLock> finalResult = new CompletableFuture<>();
        CompletableFuture<GUID> txStarted = startRawTransaction(title, TRANSACTION_TIMEOUT_SECONDS);
        txStarted.whenCompleteAsync((transactionId, startFailure) -> {
            if (startFailure != null) {
                finalResult.completeExceptionally(startFailure);
                return;
            }
            try {
                final PingedRawTransaction tx = new PingedRawTransaction(transactionId, TRANSACTION_TIMEOUT_SECONDS);
                try {
                    CompletableFuture<GUID> lockCreated = createRawLock(transactionId, path, waitable);
                    lockCreated.whenCompleteAsync((lockId, lockFailure) -> {
                        if (lockFailure != null) {
                            finalResult.completeExceptionally(lockFailure);
                            return;
                        }
                        try {
                            finalResult.complete(new LockImpl(tx, lockId));
                        } catch (Throwable unexpectedFailure) {
                            try {
                                tx.abort(unexpectedFailure);
                            } finally {
                                finalResult.completeExceptionally(unexpectedFailure);
                            }
                        }
                    }, cachedExecutor);
                } catch (Throwable createRawLockFailure) {
                    tx.abort(createRawLockFailure);
                    throw createRawLockFailure;
                }
            } catch (Throwable lockingFailure) {
                finalResult.completeExceptionally(lockingFailure);
            }
        }, cachedExecutor);
        return finalResult;
    }

    private CompletableFuture<GUID> startRawTransaction(String title, int timeoutSeconds) {
        return CompletableFuture.supplyAsync(() -> {
            Map<String, YTreeNode> attributes = new HashMap<>();
            if (title != null) {
                attributes.put("title", YTree.stringNode(title));
            }
            return yt.transactions().start(Optional.empty(), false, Duration.ofSeconds(timeoutSeconds), attributes);
        }, cachedExecutor);
    }

    private CompletableFuture<Void> pingRawTransaction(GUID transactionId) {
        return CompletableFuture.runAsync(() -> yt.transactions().ping(transactionId), cachedExecutor);
    }

    private CompletableFuture<Void> abortRawTransaction(GUID transactionId) {
        return CompletableFuture.runAsync(() -> yt.transactions().abort(transactionId), cachedExecutor);
    }

    private CompletableFuture<GUID> createRawLock(GUID transactionId, String path, boolean waitable) {
        return CompletableFuture
                .supplyAsync(() -> yt.cypress().lock(transactionId, YPath.simple(path), LockMode.EXCLUSIVE, waitable),
                        cachedExecutor);
    }

    private CompletableFuture<String> getRawLockState(GUID lockId) {
        return CompletableFuture
                .supplyAsync(() -> yt.cypress().get(YPath.objectRoot(lockId).attribute("state")).stringValue(),
                        cachedExecutor);
    }

    @Override
    public CompletableFuture<Void> remove(String path) {
        return CompletableFuture.runAsync(() -> yt.cypress().remove(YPath.simple(path)), syncExecutor);
    }

    @Override
    public CompletableFuture<Void> createTable(String path, Map<String, YTreeNode> attributes) {
        MapF<String, YTreeNode> kosherAttributes = Cf.hashMap();
        for (Map.Entry<String, YTreeNode> entry : attributes.entrySet()) {
            kosherAttributes.put(entry.getKey(), entry.getValue());
        }
        return CompletableFuture.runAsync(() -> {
            logger.info("Creating table {}", path);
            yt.cypress().create(YPath.simple(path), CypressNodeType.TABLE, true, true, kosherAttributes);
        }, syncExecutor);
    }

    @Override
    public CompletableFuture<Void> reshardTable(String path, PivotKeys pivotKeys) {
        return CompletableFuture.runAsync(() -> {
            logger.info("Resharding table {} with {} pivot keys", path, pivotKeys.size());
            yt.tables().reshard(YPath.simple(path), pivotKeys.toKosherReshardTableKeys());
        }, syncExecutor);
    }

    @Override
    public CompletableFuture<Void> mountTable(String path) {
        return CompletableFuture.runAsync(() -> {
            logger.info("Mounting table {}", path);
            yt.tables().mount(YPath.simple(path));
        }, syncExecutor);
    }

    @Override
    public CompletableFuture<Void> unfreezeTable(String path) {
        return CompletableFuture.runAsync(() -> {
            logger.info("Unfreezing table {}", path);
            yt.tables().unfreeze(YPath.simple(path));
        }, syncExecutor);
    }

    @Override
    public CompletableFuture<Void> unmountTable(String path) {
        return CompletableFuture.runAsync(() -> {
            logger.info("Unmounting table {}", path);
            yt.tables().unmount(YPath.simple(path));
        }, syncExecutor);
    }

    @Override
    public CompletableFuture<List<FlatRow>> selectRows(String query, TableSchema resultSchema) {
        return CompletableFuture.supplyAsync(() -> {
            YTSYNC_LOGGER.debug("Selecting rows: {}", query);
            List<FlatRow> result = new ArrayList<>();
            yt.tables().selectRows(query, new KosherFlatRowDeserializer(resultSchema).toEntryType(),
                    (Consumer<FlatRow>) result::add);
            return result;
        }, syncExecutor);
    }

    @Override
    public CompletableFuture<UnversionedRowset> selectRows(String query) {
        throw new UnsupportedOperationException();
    }

    @Override
    public CompletableFuture<? extends BasicTransaction> nullTransaction() {
        return CompletableFuture.completedFuture(new NullTransaction());
    }

    @Override
    public CompletableFuture<? extends BasicTransaction> startTransaction() {
        return nullTransaction();
    }

    @SuppressWarnings("unchecked")
    private static IterableF<FlatRowView> wrapRows(List<? extends FlatRowView> list) {
        // К сожалению в inside-yt нет правильных wildcard'ов
        // В данном случае cast безопасен, т.к. IterableF можно только читать
        return (IterableF<FlatRowView>) Cf.wrap(list);
    }

    /**
     * По факту через inside-yt нет транзакций для нужных нам операций
     */
    private class NullTransaction implements BasicTransaction {
        @Override
        public BasicYtSupport support() {
            return BasicYtSupportViaKosher.this;
        }

        @Override
        public boolean isAtomic() {
            return false;
        }

        @Override
        public CompletableFuture<Void> ping() {
            return CompletableFuture.completedFuture(null);
        }

        @Override
        public CompletableFuture<Void> abort() {
            return CompletableFuture.completedFuture(null);
        }

        @Override
        public CompletableFuture<Void> commit() {
            return CompletableFuture.completedFuture(null);
        }

        @Override
        public CompletableFuture<Void> insertRows(String path, TableSchema schema, List<? extends FlatRowView> rows) {
            if (rows.isEmpty()) {
                return CompletableFuture.completedFuture(null);
            }
            return CompletableFuture.runAsync(() -> {
                YTSYNC_LOGGER.info("Table {}: inserting {} rows", path, rows.size());
                yt.tables().insertRows(YPath.simple(path), false, false,
                        new KosherFlatRowSerializer<>(schema).toEntryType(),
                        wrapRows(rows).iterator(), compressorFactory.createCompressor());
            }, syncExecutor);
        }

        @Override
        public CompletableFuture<Void> updateRows(String path, TableSchema schema, List<? extends FlatRowView> rows) {
            if (rows.isEmpty()) {
                return CompletableFuture.completedFuture(null);
            }
            return CompletableFuture.runAsync(() -> {
                YTSYNC_LOGGER.info("Table {}: updating {} rows", path, rows.size());
                yt.tables().insertRows(YPath.simple(path), true, false,
                        new KosherFlatRowSerializer<>(schema).toEntryType(),
                        wrapRows(rows).iterator(), compressorFactory.createCompressor());
            }, syncExecutor);
        }

        @Override
        public CompletableFuture<Void> deleteRows(String path, TableSchema schema, List<? extends FlatRowView> keys) {
            if (keys.isEmpty()) {
                return CompletableFuture.completedFuture(null);
            }
            return CompletableFuture.runAsync(() -> {
                YTSYNC_LOGGER.info("Table {}: deleting {} rows", path, keys.size());
                yt.tables().deleteRows(YPath.simple(path), new KosherFlatRowSerializer<>(schema).toEntryType(),
                        wrapRows(keys), compressorFactory.createCompressor());
            }, syncExecutor);
        }

        @Override
        public CompletableFuture<Void> modifyRows(String path, TableSchema schema,
                                                  List<? extends FlatRowView> insertedRows, List<? extends FlatRowView> updatedRows,
                                                  List<? extends FlatRowView> deletedKeys) {
            return CompletableFuture.allOf(
                    insertRows(path, schema, insertedRows),
                    updateRows(path, schema, updatedRows),
                    deleteRows(path, schema, deletedKeys));
        }

        @Override
        public CompletableFuture<List<FlatRow>> lookupRows(String path, TableSchema keySchema,
                                                           List<? extends FlatRowView> keys, TableSchema resultSchema) {
            if (keys.isEmpty()) {
                return CompletableFuture.completedFuture(Collections.emptyList());
            }
            return CompletableFuture.supplyAsync(() -> {
                YTSYNC_LOGGER.info("Table {}: looking up {} rows", path, keys.size());
                List<FlatRow> result = new ArrayList<>(keys.size());
                yt.tables().lookupRows(YPath.simple(path),
                        new KosherFlatRowSerializer<>(keySchema).toEntryType(),
                        wrapRows(keys),
                        new KosherFlatRowDeserializer(resultSchema).toEntryType(),
                        (Consumer<FlatRow>) result::add);
                return result;
            }, syncExecutor);
        }
    }

    /**
     * Текущее состояние пигнуемой транзакции
     */
    private enum PingedRawTransactionState {
        PINGING,
        SLEEPING,
        ABORTING;
    }

    /**
     * Стейт машина, которая периодически пингует транзакцию
     */
    private class PingedRawTransaction {
        private final GUID transactionId;
        private final int timeoutSeconds;
        private final long pingPeriodNanos;
        private final CompletableFuture<Void> finished = new CompletableFuture<>();

        private final Lock lock = new ReentrantLock();
        private PingedRawTransactionState state = PingedRawTransactionState.PINGING;
        private long lastPingStartNanoTime;
        private ScheduledFuture<?> sleepingFuture;
        private CompletableFuture<Void> pingingFuture;
        private CompletableFuture<Void> abortingFuture;

        public PingedRawTransaction(GUID transactionId, int timeoutSeconds) {
            this.transactionId = transactionId;
            this.timeoutSeconds = timeoutSeconds;
            this.pingPeriodNanos = TimeUnit.SECONDS.toNanos(timeoutSeconds) / 3;
            this.lastPingStartNanoTime = System.nanoTime();
            scheduleSleep();
        }

        private void scheduleSleep() {
            lock.lock();
            try {
                if (state != PingedRawTransactionState.PINGING) {
                    // Засыпаем, только если до этого пинговали
                    return;
                }
                state = PingedRawTransactionState.SLEEPING;
                long nextPingStartNanoTime = lastPingStartNanoTime + pingPeriodNanos;
                long delayNanos = nextPingStartNanoTime - System.nanoTime();
                if (delayNanos < 0) {
                    delayNanos = 0;
                }
                sleepingFuture = executor.schedule(this::startNextPing, delayNanos, TimeUnit.NANOSECONDS);
            } finally {
                lock.unlock();
            }
        }

        private void startNextPing() {
            try {
                lock.lock();
                try {
                    if (state != PingedRawTransactionState.SLEEPING) {
                        // Пингуем, только если до этого спали
                        return;
                    }
                    state = PingedRawTransactionState.PINGING;
                    sleepingFuture = null;
                    lastPingStartNanoTime = System.nanoTime();
                    pingingFuture = pingRawTransaction(transactionId);
                    pingingFuture.whenComplete(this::pingFinished);
                } finally {
                    lock.unlock();
                }
            } catch (Throwable e) {
                abort(e);
                throw e;
            }
        }

        private void pingFinished(Void ignored, Throwable error) {
            if (error != null) {
                abort(error);
            } else {
                scheduleSleep();
            }
        }

        private CompletableFuture<Void> abort(Throwable reason) {
            lock.lock();
            try {
                Future<?> toCancel;
                PingedRawTransactionState previousState = this.state;
                switch (previousState) {
                    case PINGING:
                        // Было бы неплохо немного подождать окончания ping'а
                        toCancel = pingingFuture;
                        break;
                    case SLEEPING:
                        toCancel = sleepingFuture;
                        break;
                    default:
                        return finished;
                }
                state = PingedRawTransactionState.ABORTING;
                if (toCancel != null) {
                    toCancel.cancel(false);
                }
                abortingFuture = abortRawTransaction(transactionId);
                abortingFuture.whenComplete((r, e) -> {
                    if (!finished.isDone()) {
                        //noinspection Duplicates
                        if (reason != null) {
                            finished.completeExceptionally(reason);
                        } else if (e != null) {
                            finished.completeExceptionally(e);
                        } else {
                            finished.complete(r);
                        }
                    }
                });
                return finished;
            } finally {
                lock.unlock();
            }
        }
    }

    private class LockImpl implements YtLock {
        private final PingedRawTransaction tx;
        private final GUID lockId;

        public LockImpl(PingedRawTransaction tx, GUID lockId) {
            this.tx = tx;
            this.lockId = lockId;
        }

        @Override
        public CompletableFuture<Void> cancel() {
            return tx.abort(null);
        }

        @Override
        public CompletableFuture<Boolean> acquired() {
            return getRawLockState(lockId).thenApply("acquired"::equals);
        }

        @Override
        public CompletableFuture<Void> unlocked() {
            return tx.finished;
        }
    }
}
