package ru.yandex.direct.mysql.ytsync.common.compatibility;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import one.util.streamex.StreamEx;

import ru.yandex.direct.mysql.ytsync.common.row.FlatRow;
import ru.yandex.direct.mysql.ytsync.common.row.FlatRowView;
import ru.yandex.direct.mysql.ytsync.common.util.YtSyncCommonUtil;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;
import ru.yandex.yt.rpcproxy.ETransactionType;
import ru.yandex.yt.ytclient.proxy.ApiServiceClient;
import ru.yandex.yt.ytclient.proxy.ApiServiceTransaction;
import ru.yandex.yt.ytclient.proxy.ApiServiceTransactionOptions;
import ru.yandex.yt.ytclient.proxy.LookupRowsRequest;
import ru.yandex.yt.ytclient.proxy.ModifyRowsRequest;
import ru.yandex.yt.ytclient.proxy.request.ColumnFilter;
import ru.yandex.yt.ytclient.proxy.request.GetNode;
import ru.yandex.yt.ytclient.rpc.RpcUtil;
import ru.yandex.yt.ytclient.tables.TableSchema;
import ru.yandex.yt.ytclient.wire.UnversionedRow;
import ru.yandex.yt.ytclient.wire.UnversionedRowset;
import ru.yandex.yt.ytclient.wire.UnversionedValue;

import static ru.yandex.direct.mysql.ytsync.common.util.YtSyncCommonUtil.YTSYNC_LOGGER;

/**
 * Реализация BasicYtSupport используя Директовый ytclient
 * <p>
 * Пока нет поддержки для всех нужных операций они делегируются на вышестоящий клиент
 */
public class BasicYtSupportViaYtClient extends BasicYtSupportDelegate {
    private final ApiServiceClient client;
    private final ScheduledExecutorService executor;

    public BasicYtSupportViaYtClient(BasicYtSupport base, ApiServiceClient client, ScheduledExecutorService executor) {
        super(base);
        this.client = client;
        this.executor = executor;
    }

    @Override
    public ScheduledExecutorService executor() {
        return executor;
    }

    @Override
    public CompletableFuture<YTreeNode> getNode(String path) {
        return client.getNode(path);
    }

    @Override
    public CompletableFuture<YTreeNode> getNode(String path, Set<String> attributes) {
        return client.getNode(
                new GetNode(path)
                        .setAttributes(new ColumnFilter().setColumns(StreamEx.of(attributes).toList()))
        );
    }

    private static List<FlatRow> convertRowset(UnversionedRowset rowset, TableSchema resultSchema) {
        if (!resultSchema.getColumnNames().equals(rowset.getSchema().getColumnNames())) {
            throw new IllegalStateException(
                    "Result schema mismatch, expected " + resultSchema.getColumnNames() + ", got " + rowset.getSchema()
                            .getColumnNames());
        }
        List<FlatRow> resultRows = new ArrayList<>(rowset.getRows().size());
        for (UnversionedRow row : rowset.getRows()) {
            if (row == null) {
                // По идее такое возможно только при keepMissingRows=true
                // Конвертируем в null на случае если в будущем будем этим пользоваться
                resultRows.add(null);
                continue;
            }
            FlatRow resultRow = new FlatRow(resultSchema.getColumnsCount());
            List<UnversionedValue> values = row.getValues();
            for (int i = 0; i < resultSchema.getColumnsCount(); ++i) {
                resultRow.set(i, values.get(i).toYTree());
            }
            resultRows.add(resultRow);
        }
        return resultRows;
    }

    @Override
    public CompletableFuture<List<FlatRow>> selectRows(String query, TableSchema resultSchema) {
        YTSYNC_LOGGER.debug("Selecting rows: {}", query);
        return RpcUtil.applyAsync(client.selectRows(query), rowset -> convertRowset(rowset, resultSchema), executor);
    }

    public CompletableFuture<UnversionedRowset> selectRows(String query) {
        YTSYNC_LOGGER.debug("Selecting rows: {}", query);
        return RpcUtil.applyAsync(client.selectRows(query), t -> t, executor);
    }

    private CompletableFuture<ApiServiceTransaction> createRawTabletTransaction() {
        ApiServiceTransactionOptions options = new ApiServiceTransactionOptions(ETransactionType.TT_TABLET)
                .setSticky(true);
        return YtSyncCommonUtil.startMeasured(
                () -> YTSYNC_LOGGER.info("Creating tablet transaction..."),
                () -> client.startTransaction(options),
                (tx, ns) -> YTSYNC_LOGGER
                        .info("Created tablet transaction {} in {}ms", tx.getId(), TimeUnit.NANOSECONDS.toMillis(ns)));
    }

    @Override
    public CompletableFuture<? extends BasicYtSupport.BasicTransaction> nullTransaction() {
        return CompletableFuture.completedFuture(new NullTransaction());
    }

    @Override
    public CompletableFuture<? extends BasicYtSupport.BasicTransaction> startTransaction() {
        return RpcUtil.apply(createRawTabletTransaction(), RealTransaction::new);
    }

    public class NullTransaction implements BasicYtSupport.BasicTransaction {
        @Override
        public BasicYtSupport support() {
            return BasicYtSupportViaYtClient.this;
        }

        @Override
        public boolean isAtomic() {
            return false;
        }

        @Override
        public CompletableFuture<Void> ping() {
            return CompletableFuture.completedFuture(null);
        }

        @Override
        public CompletableFuture<Void> abort() {
            return CompletableFuture.completedFuture(null);
        }

        @Override
        public CompletableFuture<Void> commit() {
            return CompletableFuture.completedFuture(null);
        }

        @Override
        public CompletableFuture<Void> insertRows(String path, TableSchema schema, List<? extends FlatRowView> rows) {
            if (rows.isEmpty()) {
                return CompletableFuture.completedFuture(null);
            }
            return YtSupportUtil
                    .runBasicTransaction(startTransaction(), tx -> tx.insertRows(path, schema, rows), executor());
        }

        @Override
        public CompletableFuture<Void> updateRows(String path, TableSchema schema, List<? extends FlatRowView> rows) {
            if (rows.isEmpty()) {
                return CompletableFuture.completedFuture(null);
            }
            return YtSupportUtil
                    .runBasicTransaction(startTransaction(), tx -> tx.updateRows(path, schema, rows), executor());
        }

        @Override
        public CompletableFuture<Void> deleteRows(String path, TableSchema schema, List<? extends FlatRowView> keys) {
            if (keys.isEmpty()) {
                return CompletableFuture.completedFuture(null);
            }
            return YtSupportUtil
                    .runBasicTransaction(startTransaction(), tx -> tx.deleteRows(path, schema, keys), executor());
        }

        @Override
        public CompletableFuture<Void> modifyRows(String path, TableSchema schema,
                                                  List<? extends FlatRowView> insertedRows,
                                                  List<? extends FlatRowView> updatedRows,
                                                  List<? extends FlatRowView> deletedKeys) {
            if (insertedRows.isEmpty() && updatedRows.isEmpty() && deletedKeys.isEmpty()) {
                return CompletableFuture.completedFuture(null);
            }
            return YtSupportUtil.runBasicTransaction(startTransaction(),
                    tx -> tx.modifyRows(path, schema, insertedRows, updatedRows, deletedKeys), executor());
        }

        @Override
        public CompletableFuture<List<FlatRow>> lookupRows(String path, TableSchema keySchema,
                                                           List<? extends FlatRowView> keys, TableSchema resultSchema) {
            if (keys.isEmpty()) {
                return CompletableFuture.completedFuture(Collections.emptyList());
            }
            LookupRowsRequest request = new LookupRowsRequest(path, keySchema)
                    .addFilters(keys)
                    .setKeepMissingRows(false)
                    .addLookupColumns(resultSchema.getColumnNames());
            return RpcUtil.applyAsync(
                    YtSyncCommonUtil.startMeasured(
                            () -> YTSYNC_LOGGER.info("Table {}: looking up {} rows", path, keys.size()),
                            () -> client.lookupRows(request),
                            (rowset, ns) -> YTSYNC_LOGGER
                                    .info("Table {}: looked up {} rows in {}ms", path, rowset.getRows().size(),
                                            TimeUnit.NANOSECONDS.toMillis(ns))),
                    rowset -> convertRowset(rowset, resultSchema),
                    executor);
        }
    }

    public class RealTransaction implements BasicYtSupport.BasicTransaction {
        private final ApiServiceTransaction tx;

        public RealTransaction(ApiServiceTransaction tx) {
            this.tx = tx;
        }

        @Override
        public BasicYtSupport support() {
            return BasicYtSupportViaYtClient.this;
        }

        @Override
        public boolean isAtomic() {
            return true;
        }

        @Override
        public CompletableFuture<Void> ping() {
            YTSYNC_LOGGER.info("Pinging transaction {}", tx.getId());
            return tx.ping();
        }

        @Override
        public CompletableFuture<Void> abort() {
            YTSYNC_LOGGER.info("Aborting transaction {}", tx.getId());
            return tx.abort();
        }

        @Override
        public CompletableFuture<Void> commit() {
            return YtSyncCommonUtil.startMeasured(
                    () -> YTSYNC_LOGGER.info("Committing transaction {}", tx.getId()),
                    tx::commit,
                    (ignored, ns) -> YTSYNC_LOGGER
                            .info("Committed transaction {} in {}ms", tx.getId(), TimeUnit.NANOSECONDS.toMillis(ns)));
        }

        @Override
        public CompletableFuture<Void> insertRows(String path, TableSchema schema, List<? extends FlatRowView> rows) {
            if (rows.isEmpty()) {
                return CompletableFuture.completedFuture(null);
            }
            YTSYNC_LOGGER.info("Table {}: inserting {} rows", path, rows.size());
            return tx.modifyRows(new ModifyRowsRequest(path, schema).addInserts(rows));
        }

        @Override
        public CompletableFuture<Void> updateRows(String path, TableSchema schema, List<? extends FlatRowView> rows) {
            if (rows.isEmpty()) {
                return CompletableFuture.completedFuture(null);
            }
            YTSYNC_LOGGER.info("Table {}: updating {} rows", path, rows.size());
            return tx.modifyRows(new ModifyRowsRequest(path, schema).addUpdates(rows));
        }

        @Override
        public CompletableFuture<Void> deleteRows(String path, TableSchema schema, List<? extends FlatRowView> keys) {
            if (keys.isEmpty()) {
                return CompletableFuture.completedFuture(null);
            }
            YTSYNC_LOGGER.info("Table {}: deleting {} rows", path, keys.size());
            return tx.modifyRows(new ModifyRowsRequest(path, schema).addDeletes(keys));
        }

        @Override
        public CompletableFuture<Void> modifyRows(String path, TableSchema schema,
                                                  List<? extends FlatRowView> insertedRows,
                                                  List<? extends FlatRowView> updatedRows,
                                                  List<? extends FlatRowView> deletedKeys) {
            if (insertedRows.isEmpty() && updatedRows.isEmpty() && deletedKeys.isEmpty()) {
                return CompletableFuture.completedFuture(null);
            }
            YTSYNC_LOGGER.info("Table {}: modifying {} rows", path,
                    insertedRows.size() + updatedRows.size() + deletedKeys.size());
            return tx.modifyRows(new ModifyRowsRequest(path, schema)
                    .addInserts(insertedRows)
                    .addUpdates(updatedRows)
                    .addDeletes(deletedKeys));
        }

        @Override
        public CompletableFuture<List<FlatRow>> lookupRows(String path, TableSchema keySchema,
                                                           List<? extends FlatRowView> keys, TableSchema resultSchema) {
            if (keys.isEmpty()) {
                return CompletableFuture.completedFuture(Collections.emptyList());
            }
            LookupRowsRequest request = new LookupRowsRequest(path, keySchema)
                    .addFilters(keys)
                    .setKeepMissingRows(false)
                    .addLookupColumns(resultSchema.getColumnNames());
            return RpcUtil.applyAsync(
                    YtSyncCommonUtil.startMeasured(
                            () -> YTSYNC_LOGGER.info("Table {}: looking up {} rows", path, keys.size()),
                            () -> tx.lookupRows(request),
                            (rowset, ns) -> YTSYNC_LOGGER
                                    .info("Table {}: looked up {} rows in {}ms", path, rowset.getRows().size(),
                                            TimeUnit.NANOSECONDS.toMillis(ns))),
                    rowset -> convertRowset(rowset, resultSchema),
                    executor());
        }
    }
}
