package ru.yandex.direct.mysql.ytsync.common.compatibility;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;

import ru.yandex.direct.mysql.ytsync.common.row.FlatRow;
import ru.yandex.direct.mysql.ytsync.common.row.FlatRowView;
import ru.yandex.direct.mysql.ytsync.common.util.YtSyncCommonUtil;
import ru.yandex.yt.ytclient.tables.TableSchema;

/**
 * Нарезает операции на более мелкие кусочки
 */
public class BasicYtSupportWithChunking extends BasicYtSupportDelegate {
    private final int maxChunkSize;

    public BasicYtSupportWithChunking(BasicYtSupport base, int maxChunkSize) {
        super(base);
        this.maxChunkSize = maxChunkSize;
    }

    private CompletableFuture<List<FlatRow>> gatherRows(List<CompletableFuture<List<FlatRow>>> ops) {
        return YtSyncCommonUtil.allFutures(ops).thenApplyAsync(ignored -> {
            List<FlatRow> result = new ArrayList<>();
            for (CompletableFuture<List<FlatRow>> op : ops) {
                result.addAll(op.getNow(null));
            }
            return result;
        }, executor());
    }

    @Override
    protected BasicTransaction wrapTransaction(BasicTransaction tx) {
        return new WrappedTransaction(tx);
    }

    public class WrappedTransaction implements BasicTransaction {
        private final BasicTransaction tx;

        public WrappedTransaction(BasicTransaction tx) {
            this.tx = tx;
        }

        @Override
        public BasicYtSupport support() {
            return BasicYtSupportWithChunking.this;
        }

        @Override
        public boolean isAtomic() {
            return tx.isAtomic();
        }

        @Override
        public CompletableFuture<Void> ping() {
            return tx.ping();
        }

        @Override
        public CompletableFuture<Void> abort() {
            return tx.abort();
        }

        @Override
        public CompletableFuture<Void> commit() {
            return tx.commit();
        }

        @Override
        public CompletableFuture<Void> insertRows(String path, TableSchema schema, List<? extends FlatRowView> rows) {
            if (rows.size() <= maxChunkSize) {
                return tx.insertRows(path, schema, rows);
            }
            List<CompletableFuture<Void>> ops = new ArrayList<>();
            for (int start = 0; start < rows.size(); start += maxChunkSize) {
                int end = Math.min(start + maxChunkSize, rows.size());
                ops.add(tx.insertRows(path, schema, rows.subList(start, end)));
            }
            return YtSyncCommonUtil.allFutures(ops);
        }

        @Override
        public CompletableFuture<Void> updateRows(String path, TableSchema schema, List<? extends FlatRowView> rows) {
            if (rows.size() <= maxChunkSize) {
                return tx.updateRows(path, schema, rows);
            }
            List<CompletableFuture<Void>> ops = new ArrayList<>();
            for (int start = 0; start < rows.size(); start += maxChunkSize) {
                int end = Math.min(start + maxChunkSize, rows.size());
                ops.add(tx.updateRows(path, schema, rows.subList(start, end)));
            }
            return YtSyncCommonUtil.allFutures(ops);
        }

        @Override
        public CompletableFuture<Void> deleteRows(String path, TableSchema schema, List<? extends FlatRowView> keys) {
            if (keys.size() <= maxChunkSize) {
                return tx.deleteRows(path, schema, keys);
            }
            List<CompletableFuture<Void>> ops = new ArrayList<>();
            for (int start = 0; start < keys.size(); start += maxChunkSize) {
                int end = Math.min(start + maxChunkSize, keys.size());
                ops.add(tx.deleteRows(path, schema, keys.subList(start, end)));
            }
            return YtSyncCommonUtil.allFutures(ops);
        }

        @Override
        public CompletableFuture<Void> modifyRows(String path, TableSchema schema,
                                                  List<? extends FlatRowView> insertedRows, List<? extends FlatRowView> updatedRows,
                                                  List<? extends FlatRowView> deletedKeys) {
            if (insertedRows.size() + updatedRows.size() + deletedKeys.size() <= maxChunkSize) {
                return tx.modifyRows(path, schema, insertedRows, updatedRows, deletedKeys);
            }
            return CompletableFuture.allOf(
                    insertRows(path, schema, insertedRows),
                    updateRows(path, schema, updatedRows),
                    deleteRows(path, schema, deletedKeys));
        }

        @Override
        public CompletableFuture<List<FlatRow>> lookupRows(String path, TableSchema keySchema,
                                                           List<? extends FlatRowView> keys,
                                                           TableSchema resultSchema) {
            if (keys.size() <= maxChunkSize) {
                return tx.lookupRows(path, keySchema, keys, resultSchema);
            }
            final List<CompletableFuture<List<FlatRow>>> ops = new ArrayList<>();
            for (int start = 0; start < keys.size(); start += maxChunkSize) {
                int end = Math.min(start + maxChunkSize, keys.size());
                ops.add(tx.lookupRows(path, keySchema, keys.subList(start, end), resultSchema));
            }
            return gatherRows(ops);
        }
    }
}
