package ru.yandex.direct.mysql.ytsync.common.compatibility;

import java.time.Duration;
import java.util.List;
import java.util.concurrent.CompletableFuture;

import ru.yandex.direct.mysql.ytsync.common.row.FlatRow;
import ru.yandex.direct.mysql.ytsync.common.row.FlatRowView;
import ru.yandex.yt.ytclient.tables.TableSchema;

/**
 * Добавляет к операциям автоматические повторы в случае ошибок
 */
public class BasicYtSupportWithRetries extends BasicYtSupportDelegate {
    private final Duration innerTimeout;
    private final Duration outerTimeout;

    public BasicYtSupportWithRetries(BasicYtSupport base, Duration innerTimeout, Duration outerTimeout) {
        super(base);
        this.innerTimeout = innerTimeout;
        this.outerTimeout = outerTimeout;
    }

    @Override
    public CompletableFuture<List<FlatRow>> selectRows(String query, TableSchema resultSchema) {
        return OperationWithRetries
                .start(() -> base().selectRows(query, resultSchema), executor(), outerTimeout, false);
    }

    @Override
    public CompletableFuture<? extends BasicTransaction> startTransaction() {
        return OperationWithRetries.start(super::startTransaction, executor(), outerTimeout, false);
    }

    @Override
    protected BasicTransaction wrapTransaction(BasicTransaction tx) {
        return new WrappedTransaction(tx);
    }

    public class WrappedTransaction implements BasicTransaction {
        private final BasicTransaction tx;

        public WrappedTransaction(BasicTransaction tx) {
            this.tx = tx;
        }

        @Override
        public BasicYtSupport support() {
            return BasicYtSupportWithRetries.this;
        }

        @Override
        public boolean isAtomic() {
            return tx.isAtomic();
        }

        @Override
        public CompletableFuture<Void> ping() {
            return tx.ping();
        }

        @Override
        public CompletableFuture<Void> abort() {
            return tx.abort();
        }

        @Override
        public CompletableFuture<Void> commit() {
            return tx.commit();
        }

        @Override
        public CompletableFuture<Void> insertRows(String path, TableSchema schema, List<? extends FlatRowView> rows) {
            return OperationWithRetries.start(() -> tx.insertRows(path, schema, rows), executor(),
                    isAtomic() ? innerTimeout : outerTimeout, isAtomic());
        }

        @Override
        public CompletableFuture<Void> updateRows(String path, TableSchema schema, List<? extends FlatRowView> rows) {
            return OperationWithRetries.start(() -> tx.updateRows(path, schema, rows), executor(),
                    isAtomic() ? innerTimeout : outerTimeout, isAtomic());
        }

        @Override
        public CompletableFuture<Void> deleteRows(String path, TableSchema schema, List<? extends FlatRowView> keys) {
            return OperationWithRetries.start(() -> tx.deleteRows(path, schema, keys), executor(),
                    isAtomic() ? innerTimeout : outerTimeout, isAtomic());
        }

        @Override
        public CompletableFuture<Void> modifyRows(String path, TableSchema schema,
                                                  List<? extends FlatRowView> insertedRows, List<? extends FlatRowView> updatedRows,
                                                  List<? extends FlatRowView> deletedKeys) {
            return OperationWithRetries
                    .start(() -> tx.modifyRows(path, schema, insertedRows, updatedRows, deletedKeys), executor(),
                            isAtomic() ? innerTimeout : outerTimeout, isAtomic());
        }

        @Override
        public CompletableFuture<List<FlatRow>> lookupRows(String path, TableSchema keySchema,
                                                           List<? extends FlatRowView> keys,
                                                           TableSchema resultSchema) {
            return OperationWithRetries
                    .start(() -> tx.lookupRows(path, keySchema, keys, resultSchema), executor(),
                            isAtomic() ? innerTimeout : outerTimeout, isAtomic());
        }
    }
}
