package ru.yandex.direct.mysql.ytsync.common.compatibility;

import java.time.Duration;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.function.Supplier;

/**
 * Асинхронная проверка условия раз в определённый период
 */
public class OperationWaitCondition {
    private final CompletableFuture<Void> resultFuture;
    private final Supplier<CompletableFuture<Boolean>> supplier;
    private final ScheduledExecutorService executor;
    private final Runnable beforeWaitCallback;
    private final long period;
    private final long deadline;
    private long currentStartTime;
    private CompletableFuture<Boolean> currentResult;
    private ScheduledFuture<?> currentTimer;

    private OperationWaitCondition(CompletableFuture<Void> resultFuture, Supplier<CompletableFuture<Boolean>> supplier,
                                   ScheduledExecutorService executor, Runnable beforeWaitCallback, Duration period, Duration timeout) {
        this.resultFuture = resultFuture;
        this.supplier = supplier;
        this.executor = executor;
        this.beforeWaitCallback = beforeWaitCallback;
        this.period = period.toNanos();
        this.deadline = System.nanoTime() + timeout.toNanos();
    }

    /**
     * Вызывается когда потребителя перестаёт интересовать результат операции
     */
    private synchronized void discard() {
        if (currentResult != null) {
            if (!currentResult.isDone()) {
                // Внимание: возможен рекурсивный вызов processResult
                currentResult.cancel(false);
            }
            currentResult = null;
        }
        if (currentTimer != null) {
            currentTimer.cancel(false);
            currentTimer = null;
        }
    }

    private synchronized void nextAttempt() {
        currentTimer = null;
        if (resultFuture.isDone()) {
            return;
        }
        try {
            currentStartTime = System.nanoTime();
            currentResult = supplier.get();
            currentResult.whenComplete(this::processResult);
        } catch (Throwable e) {
            resultFuture.completeExceptionally(e);
        }
    }

    private synchronized void processResult(Boolean result, Throwable exception) {
        currentResult = null;
        if (resultFuture.isDone()) {
            return;
        }
        try {
            if (exception != null) {
                resultFuture.completeExceptionally(exception);
            } else if (result) {
                resultFuture.complete(null);
            } else {
                long nextTime = currentStartTime + period;
                if (nextTime >= deadline) {
                    resultFuture.completeExceptionally(new TimeoutException("Deadline reached"));
                    return;
                }
                long delay = nextTime - System.nanoTime();
                if (delay < 0) {
                    delay = 0;
                }
                if (beforeWaitCallback != null) {
                    beforeWaitCallback.run();
                }
                currentTimer = executor.schedule(this::nextAttempt, delay, TimeUnit.NANOSECONDS);
            }
        } catch (Throwable e) {
            resultFuture.completeExceptionally(e);
        }
    }

    public static CompletableFuture<Void> start(Supplier<CompletableFuture<Boolean>> supplier,
                                                ScheduledExecutorService executor, Duration period, Duration timeout, Runnable beforeWaitCallback) {
        CompletableFuture<Void> resultFuture = new CompletableFuture<>();
        OperationWaitCondition op =
                new OperationWaitCondition(resultFuture, supplier, executor, beforeWaitCallback, period, timeout);
        resultFuture.whenComplete((ignoredResult, ignoredException) -> op.discard());
        op.nextAttempt();
        return resultFuture;
    }

    public static CompletableFuture<Void> start(Supplier<CompletableFuture<Boolean>> supplier,
                                                ScheduledExecutorService executor, Duration period, Duration timeout) {
        return start(supplier, executor, period, timeout, null);
    }
}
