package ru.yandex.direct.mysql.ytsync.common.compatibility;

import java.io.IOException;
import java.time.Duration;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.function.Supplier;
import java.util.regex.Pattern;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.inside.yt.kosher.impl.common.YtException;
import ru.yandex.misc.io.RuntimeIoException;
import ru.yandex.yt.TError;
import ru.yandex.yt.ytclient.rpc.RpcError;

/**
 * Обработка цикла повтора операций с прогрессивными задержками
 */
public class OperationWithRetries<T> {
    private static final Logger logger = LoggerFactory.getLogger(OperationWithRetries.class);

    // yt/ytlib/transaction_client/public.h
    private static final int NO_SUCH_TRANSACTION = 11000;

    // Сообщение об ошибке в случае пропадания транзакции с сервера
    private static final Pattern RE_TRANSACTION_EXPIRED = Pattern.compile("Sticky transaction [^\\s]+ is not found");

    // Сообщение об ошибке в случае, если в yt client нет доступных rpc proxy
    private static final String EMPTY_CLIENT_LIST = "empty client list";

    // Прогрессивная шкала задержек между попытками, в миллисекундах
    private static final long[] delayMillis = new long[]{
            50,
            100,
            250,
            500,
            750,
            1000,
            2000,
            5000,
            10000,
            20000,
            30000
    };

    private final CompletableFuture<T> resultFuture;
    private final Supplier<CompletableFuture<T>> supplier;
    private final ScheduledExecutorService executor;
    private final long deadline;
    private final boolean isAtomic;
    private long currentStartTime;
    private CompletableFuture<T> currentFuture;
    private ScheduledFuture<?> currentTimer;
    private long currentDelayMillis;
    private int retryIndex;

    private OperationWithRetries(CompletableFuture<T> resultFuture, Supplier<CompletableFuture<T>> supplier,
                                 ScheduledExecutorService executor, Duration timeout, boolean isAtomic) {
        this.supplier = Objects.requireNonNull(supplier);
        this.resultFuture = resultFuture;
        this.executor = executor;
        this.deadline = System.nanoTime() + timeout.toNanos();
        this.isAtomic = isAtomic;
    }

    /**
     * Вызывается в том случае, если потребителя перестаёт интересовать продолжение операции
     * <p>
     * Вызывается рекурсивно в случае любого завершения или отмены
     */
    private synchronized void discard() {
        if (currentFuture != null) {
            if (!currentFuture.isDone()) {
                // Внимание: может рекурсивно вызывать метод processResult ниже
                currentFuture.cancel(false);
            }
            currentFuture = null;
        }
        if (currentTimer != null) {
            currentTimer.cancel(false);
            currentTimer = null;
        }
    }

    /**
     * Проверяет возможность повтора операции после исключения e
     */
    private boolean mayRetry(Throwable e) {
        while ((e instanceof CompletionException || e instanceof ExecutionException) && e.getCause() != null) {
            e = e.getCause();
        }
        if (e instanceof IllegalStateException) {
            if (EMPTY_CLIENT_LIST.equals(e.getMessage())) {
                // Эту ошибку можно ретраить: связность с yt может вернуться
                return true;
            }
        }
        if (e instanceof YtException || e instanceof RuntimeIoException) {
            // Любые исключения в kosher реализации считаются временными :-/
            return true;
        }
        if (e instanceof TimeoutException) {
            // В случае таймаутов операции повторяются
            return true;
        }
        if (e instanceof RpcError) {
            TError error = ((RpcError) e).getError();
            if (isAtomic) {
                if (error.getCode() == NO_SUCH_TRANSACTION || RE_TRANSACTION_EXPIRED.matcher(e.getMessage()).find()) {
                    // Если транзакция пропала на сервере, то повторять операцию бесполезно
                    return false;
                }
            }
            // Пока считаем, что все остальные ошибки являются временными
            return true;
        }
        if (e instanceof IOException) {
            // Считаем любые IO ошибки временными проблемами с соединением
            return true;
        }
        // Все остальные ошибки считаем проблемами в коде
        return false;
    }

    /**
     * Запускает очередную попытку выполнения асинхронной операции
     */
    private synchronized void nextAttempt() {
        currentTimer = null;
        if (resultFuture.isDone()) {
            // Потребителя больше не интересует результат операции
            return;
        }
        try {
            currentStartTime = System.nanoTime();
            currentFuture = supplier.get();
            currentFuture.whenComplete(this::processResult);
        } catch (Throwable e) {
            logger.error("Operation failed to start", e);
            if (!resultFuture.isDone()) {
                resultFuture.completeExceptionally(e);
            }
        }
    }

    /**
     * Обрабатывает результат завершения текущей попытки
     */
    private synchronized void processResult(T result, Throwable exception) {
        currentFuture = null;
        if (resultFuture.isDone()) {
            // Потребителя больше не интересует результат операции
            return;
        }
        try {
            if (exception != null) {
                if (mayRetry(exception)) {
                    long nextDelayMillis = delayMillis[retryIndex++];
                    if (retryIndex >= delayMillis.length) {
                        retryIndex = delayMillis.length - 1;
                    }
                    // Спим случайное время между текущей и следующей задержкой, начиная от времени последней попытки
                    long nextRetryTime = ThreadLocalRandom.current().nextLong(
                            currentStartTime + TimeUnit.MILLISECONDS.toNanos(currentDelayMillis),
                            currentStartTime + TimeUnit.MILLISECONDS.toNanos(nextDelayMillis) + 1);
                    if (nextRetryTime < deadline) {
                        currentDelayMillis = nextDelayMillis;
                        long delay = nextRetryTime - System.nanoTime();
                        if (delay < 0) {
                            delay = 0;
                        }
                        logger.error("Operation failed, but will be retried in " + TimeUnit.NANOSECONDS.toMillis(delay)
                                + "ms", exception);
                        currentTimer = executor.schedule(this::nextAttempt, delay, TimeUnit.NANOSECONDS);
                        return;
                    }
                }
                logger.error("Operation failed and will not be retried", exception);
                resultFuture.completeExceptionally(exception);
            } else {
                resultFuture.complete(result);
            }
        } catch (Throwable e) {
            // Любые проблемы в обработчике завершают операцию с ошибкой
            resultFuture.completeExceptionally(e);
        }
    }

    /**
     * Стартует цикл повтора операций в случае ошибок
     */
    public static <T> CompletableFuture<T> start(Supplier<CompletableFuture<T>> supplier,
                                                 ScheduledExecutorService executor, Duration timeout, boolean isAtomic) {
        CompletableFuture<T> resultFuture = new CompletableFuture<>();
        OperationWithRetries<T>
                op = new OperationWithRetries<>(resultFuture, supplier, executor, timeout, isAtomic);
        resultFuture.whenComplete((ignoredResult, ignoredException) -> op.discard());
        op.nextAttempt();
        return resultFuture;
    }
}
