package ru.yandex.direct.mysql.ytsync.common.compatibility;

import java.time.Duration;
import java.util.Optional;
import java.util.function.BooleanSupplier;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.utils.InterruptedRuntimeException;
import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.inside.yt.kosher.common.GUID;
import ru.yandex.inside.yt.kosher.cypress.LockMode;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.impl.transactions.utils.YtTransactionsUtils;
import ru.yandex.inside.yt.kosher.transactions.Transaction;

public class YtLockUtil {
    private static final Duration ACQUIRE_CHECK_TIMEOUT = Duration.ofSeconds(3);
    private static final Logger logger = LoggerFactory.getLogger(YtLockUtil.class);

    private YtLockUtil() {
    }

    public static void runInLock(Yt yt, YPath lockPath, BooleanSupplier interruptedConsumer,
                                 TransactionalConsumer transactionConsumer) {
        YtTransactionsUtils.withTransaction(
                yt,
                Duration.ofMinutes(1),
                Optional.of(Duration.ofSeconds(15)),
                transaction -> {
                    runInLockInternal(yt, lockPath, transaction, interruptedConsumer, transactionConsumer);
                    return true;
                }
        );
    }

    private static void runInLockInternal(Yt yt, YPath lockPath, Transaction transaction,
                                          BooleanSupplier interruptedSupplier,
                                          TransactionalConsumer transactionConsumer) {
        logger.info("Trying to get lock {}", lockPath);
        GUID lockId = yt.cypress().lock(transaction.getId(), lockPath, LockMode.EXCLUSIVE, true);

        boolean acquired = false;
        while (!acquired) {
            if (interruptedSupplier.getAsBoolean()) {
                logger.info("Transaction interrupted by consumer");
                throw new InterruptedRuntimeException("Thread was interrupted");
            }

            String state = yt.cypress().get(YPath.objectRoot(lockId).attribute("state")).stringValue();
            acquired = "acquired".equals(state);
            if (!acquired) {
                try {
                    Thread.sleep(ACQUIRE_CHECK_TIMEOUT.toMillis());
                } catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                    logger.info("Thread interrupted");
                    throw new InterruptedRuntimeException("Thread was interrupted", e);
                }
            }
        }
        logger.info("Acquired lock with id {}", lockId);

        try {
            transactionConsumer.accept(transaction);
        } catch (Exception e) {
            throw new RuntimeException("Got an error while running code", e);
        }
    }

    public interface TransactionalConsumer {
        void accept(Transaction transaction) throws Exception;
    }
}
