package ru.yandex.direct.mysql.ytsync.common.compatibility;

import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.function.Function;

import org.apache.commons.lang3.tuple.Pair;

import ru.yandex.direct.mysql.ytsync.common.keys.PivotKeys;
import ru.yandex.direct.ytwrapper.model.attributes.OptimizeForAttr;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;
import ru.yandex.yt.ytclient.tables.TableSchema;

/**
 * Абстракция над различными клиентами yt
 */
public interface YtSupport extends BasicYtSupport {
    @Override
    CompletableFuture<? extends Transaction> nullTransaction();

    @Override
    CompletableFuture<? extends Transaction> startTransaction();

    /**
     * Выполняет асинхронный код внутри транзакции
     */
    <T> CompletableFuture<T> runTransaction(Function<? super Transaction, ? extends CompletableFuture<T>> fn);

    /**
     * Возвращает схему таблицы по пути path
     */
    CompletableFuture<TableSchema> getTableSchema(String path);

    /**
     * Возвращает схему и флаг isDynamic таблицы
     */
    CompletableFuture<Pair<TableSchema, Boolean>> getTableSchemaAndIsDynamic(String path);

    /**
     * Возвращает ключи шардирования таблицы по пути path
     */
    CompletableFuture<PivotKeys> getTablePivotKeys(String path);

    /**
     * Возвращает true, если все таблеты таблицы по пути path находятся в состоянии state
     */
    CompletableFuture<Boolean> areTabletStatesEqual(String path, String state);

    CompletableFuture<String> getTabletState(String path);

    /**
     * Ожидает когда таблеты таблицы path перейдут в состояние state
     */
    CompletableFuture<Void> waitTabletStates(String path, String state);

    /**
     * Возвращает true, если таблица примонтирована
     */
    CompletableFuture<Boolean> isTableMounted(String path);

    /**
     * Ожидает завершения монтирования таблицы
     */
    CompletableFuture<Void> waitTableMounted(String path);

    /**
     * Возвращает true, если таблица размонтирована
     */
    CompletableFuture<Boolean> isTableUnmounted(String path);

    /**
     * Ожидает завершения размонтирования таблицы
     */
    CompletableFuture<Void> waitTableUnmounted(String path);

    /**
     * Возвращает true, если таблица path динамическая
     */
    CompletableFuture<Boolean> isTableDynamic(String path);

    /**
     * Шардирует таблицу по ключам pivotKeys и монтирует её
     */
    CompletableFuture<Void> maybeReshardAndMount(String path, PivotKeys pivotKeys);

    /**
     * Типичный цикл подготовки динамической таблицы
     */
    CompletableFuture<Void> prepareDynamicTable(String path, TableSchema schema, OptimizeForAttr optimizeFor,
                                                PivotKeys pivotKeys, Map<String, YTreeNode> attributes);

    /**
     * Абстракция над операциями в транзакциях
     */
    interface Transaction extends BasicTransaction {
        @Override
        YtSupport support();
    }
}
