package ru.yandex.direct.mysql.ytsync.common.compatibility;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.function.Function;

import ru.yandex.yt.ytclient.proxy.ApiServiceTransaction;

public class YtSupportUtil {
    /**
     * Аналог future.thenComposeAsync(fn, executor), но не позволяет отменить результат операции
     */
    public static <T, R> CompletableFuture<R> composeWithoutCancellation(CompletableFuture<T> future,
                                                                         Function<T, CompletableFuture<R>> fn, Executor executor) {
        CompletableFuture<R> finalResult = new CompletableFuture<>();
        future.whenCompleteAsync((futureResult, futureError) -> {
            if (futureError != null) {
                finalResult.completeExceptionally(futureError);
            } else {
                try {
                    fn.apply(futureResult).whenComplete((fnResult, fnError) -> {
                        if (fnError != null) {
                            finalResult.completeExceptionally(fnError);
                        } else {
                            finalResult.complete(fnResult);
                        }
                    });
                } catch (Throwable fnError) {
                    finalResult.completeExceptionally(fnError);
                }
            }
        }, executor);
        return finalResult;
    }

    public interface AbstractTx<Tx> {
        Tx getTransaction();

        CompletableFuture<Void> abort();

        CompletableFuture<Void> commit();
    }

    public static class TxFromApiService implements AbstractTx<ApiServiceTransaction> {
        private final ApiServiceTransaction tx;

        public TxFromApiService(ApiServiceTransaction tx) {
            this.tx = tx;
        }

        @Override
        public ApiServiceTransaction getTransaction() {
            return tx;
        }

        @Override
        public CompletableFuture<Void> abort() {
            return tx.abort();
        }

        @Override
        public CompletableFuture<Void> commit() {
            return tx.commit();
        }
    }

    public static class TxFromBasicSupport<Tx extends BasicYtSupport.BasicTransaction> implements AbstractTx<Tx> {
        private final Tx tx;

        public TxFromBasicSupport(Tx tx) {
            this.tx = tx;
        }

        @Override
        public Tx getTransaction() {
            return tx;
        }

        @Override
        public CompletableFuture<Void> abort() {
            return tx.abort();
        }

        @Override
        public CompletableFuture<Void> commit() {
            return tx.commit();
        }
    }

    /**
     * Возвращает результат fn.apply(tx) и в зависимости от результата вызывает tx.commit() или tx.abort()
     */
    public static <Tx, R> CompletableFuture<R> runWithAbstractTransaction(AbstractTx<Tx> tx,
                                                                          Function<? super Tx, ? extends CompletableFuture<R>> fn) {
        CompletableFuture<R> finalResult = new CompletableFuture<>();
        try {
            fn.apply(tx.getTransaction()).whenComplete((fnResult, fnException) -> {
                try {
                    // Делаем commit или abort в зависимости от успеха операции
                    CompletableFuture<Void> commitOrAbort;
                    if (fnException != null) {
                        commitOrAbort = tx.abort();
                    } else {
                        commitOrAbort = tx.commit();
                    }
                    commitOrAbort.whenComplete((commitResult, commitException) -> {
                        if (fnException != null) {
                            // Нам не важно с каким статусом завершился abort, отдаём оригинальное исключение
                            finalResult.completeExceptionally(fnException);
                        } else if (commitException != null) {
                            // Результат был успешным, но commit провалился
                            finalResult.completeExceptionally(commitException);
                        } else {
                            // Коммит был успешным, отдаём результат
                            finalResult.complete(fnResult);
                        }
                    });
                } catch (Throwable commitException) {
                    // Вызов commit или abort кинул синхронное исключение
                    finalResult.completeExceptionally(commitException);
                }
            });
        } catch (Throwable fnException) {
            // Вызов fn кинул синхронное исключение
            try {
                tx.abort().whenComplete((abortResult, abortException) -> {
                    // Нам не важно с каким статусом завершился вызов abort
                    finalResult.completeExceptionally(fnException);
                });
            } catch (Throwable abortException) {
                // Вызов abort тоже кинул исключение
                finalResult.completeExceptionally(fnException);
            }
        }
        return finalResult;
    }

    /**
     * Дожидается получения транзакции из future, выполняет fn.apply(tx) и
     * в зависимости от результата делает commit или abort.
     */
    public static <Tx, R> CompletableFuture<R> runTransaction(CompletableFuture<Tx> future,
                                                              Function<Tx, AbstractTx<Tx>> converter, Function<? super Tx, ? extends CompletableFuture<R>> fn,
                                                              Executor executor) {
        return composeWithoutCancellation(future, tx -> runWithAbstractTransaction(converter.apply(tx), fn), executor);
    }

    /**
     * Дожидается получения транзакции из future, выполняет fn.apply(tx) и
     * в зависимости от результата делает commit или abort.
     */
    public static <R> CompletableFuture<R> runApiTransaction(CompletableFuture<ApiServiceTransaction> future,
                                                             Function<? super ApiServiceTransaction, ? extends CompletableFuture<R>> fn, Executor executor) {
        return runTransaction(future, TxFromApiService::new, fn, executor);
    }

    /**
     * Дожидается получения транзакции из future, выполняет fn.apply(tx) и
     * в зависимости от результата делает commit или abort.
     */
    public static <Tx extends BasicYtSupport.BasicTransaction, R> CompletableFuture<R> runBasicTransaction(
            CompletableFuture<Tx> future, Function<? super Tx, ? extends CompletableFuture<R>> fn, Executor executor) {
        return runTransaction(future, TxFromBasicSupport::new, fn, executor);
    }
}
