package ru.yandex.direct.mysql.ytsync.common.compatibility;

import java.time.Duration;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.function.Function;

import com.google.common.collect.ImmutableSet;
import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.mysql.ytsync.common.keys.PivotKeys;
import ru.yandex.direct.ytwrapper.YtUtils;
import ru.yandex.direct.ytwrapper.model.attributes.OptimizeForAttr;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;
import ru.yandex.yt.ytclient.tables.TableSchema;

import static java.util.stream.Collectors.toList;

/**
 * Базовая реализация YtSupport использующая BasicYtSupport
 */
public class YtSupportViaBasic extends YtSupportDelegateToBasic {
    private static final Logger logger = LoggerFactory.getLogger(YtSupportViaBasic.class);

    /**
     * Период проверок для ожидания состояния таблицы
     */
    public static final Duration STATE_WAIT_PERIOD = Duration.ofSeconds(1);
    /**
     * Таймаут на ожидание нужного состояния таблицы
     */
    public static final Duration STATE_WAIT_TIMEOUT = Duration.ofSeconds(60);

    public YtSupportViaBasic(BasicYtSupport base) {
        super(base);
    }

    @Override
    public CompletableFuture<TableSchema> getTableSchema(String path) {
        return getNode(path + "/@schema").thenApply(TableSchema::fromYTree);
    }

    @Override
    public CompletableFuture<Pair<TableSchema, Boolean>> getTableSchemaAndIsDynamic(String path) {
        return getNode(path, ImmutableSet.of(YtUtils.SCHEMA_ATTR, "dynamic")).thenApply(yTreeNode -> {
            TableSchema schema = TableSchema.fromYTree(yTreeNode.getAttributeOrThrow(YtUtils.SCHEMA_ATTR));
            boolean isDynamic = yTreeNode.getAttributeOrThrow("dynamic").boolValue();
            return Pair.of(schema, isDynamic);
        });
    }

    @Override
    public CompletableFuture<PivotKeys> getTablePivotKeys(String path) {
        return getNode(path + "/@pivot_keys").thenApply(pivotKeys ->
                new PivotKeys(pivotKeys.asList().stream().map(YTreeNode::listNode).collect(toList())));
    }

    @Override
    public CompletableFuture<Boolean> areTabletStatesEqual(String path, String state) {
        return getNode(path + "/@tablet_state").thenApply(tabletState -> state.equals(tabletState.stringValue()));
    }

    @Override
    public CompletableFuture<String> getTabletState(String path) {
        return getNode(path + "/@tablet_state").thenApply(YTreeNode::stringValue);
    }

    @Override
    public CompletableFuture<Void> waitTabletStates(String path, String state) {
        return OperationWaitCondition
                .start(() -> areTabletStatesEqual(path, state), executor(), STATE_WAIT_PERIOD, STATE_WAIT_TIMEOUT,
                        () -> logger.info("Waiting for table {} to become {}", path, state));
    }

    @Override
    public CompletableFuture<Boolean> isTableMounted(String path) {
        return areTabletStatesEqual(path, "mounted");
    }

    @Override
    public CompletableFuture<Void> waitTableMounted(String path) {
        return waitTabletStates(path, "mounted");
    }

    @Override
    public CompletableFuture<Boolean> isTableUnmounted(String path) {
        return areTabletStatesEqual(path, "unmounted");
    }

    @Override
    public CompletableFuture<Void> waitTableUnmounted(String path) {
        return waitTabletStates(path, "unmounted");
    }

    @Override
    public CompletableFuture<Boolean> isTableDynamic(String path) {
        return getNode(path + "/@dynamic").thenApply(YTreeNode::boolValue);
    }

    @Override
    public CompletableFuture<Void> maybeReshardAndMount(String path, PivotKeys pivotKeys) {
        return isTableMounted(path)
                .thenCompose(mounted -> {
                    if (pivotKeys == null) {
                        // Игнорируем ключи шардирования
                        return CompletableFuture.completedFuture(mounted);
                    }
                    return getTablePivotKeys(path).thenCompose(currentPivotKeys -> {
                        if (pivotKeys.equals(currentPivotKeys)) {
                            // Ключи таблицы уже соответствуют заданным
                            return CompletableFuture.completedFuture(mounted);
                        }
                        // Размонтируем таблицу, дожидаемся завершения, решардируем и возвращаем false
                        return (mounted ? unmountTable(path) : CompletableFuture.completedFuture(null))
                                .thenCompose(ignored -> waitTableUnmounted(path))
                                .thenCompose(ignored -> reshardTable(path, pivotKeys))
                                .thenApply(ignored -> false);
                    });
                })
                .thenCompose(mounted -> {
                    if (mounted) {
                        // Таблица уже примонтирована
                        return CompletableFuture.completedFuture(null);
                    }
                    // Если по результатам предыдущей операции таблица не примонтирована, то запускаем монтирование
                    return mountTable(path).thenCompose(ignored -> waitTableMounted(path));
                });
    }

    @Override
    public CompletableFuture<Void> prepareDynamicTable(String path, TableSchema schema, OptimizeForAttr optimizeFor,
                                                       PivotKeys pivotKeys, Map<String, YTreeNode> attributes) {
        return exists(path)
                .thenCompose(exists -> {
                    if (exists) {
                        // Таблица уже существует
                        return CompletableFuture.completedFuture(null);
                    }
                    return createDynamicTable(path, schema, optimizeFor, attributes);
                })
                // Проверяем, что таблица динамическая
                .thenCompose(ignored -> isTableDynamic(path))
                .thenAccept(dynamic -> {
                    if (!dynamic) {
                        throw new IllegalStateException("Table " + path + " is not dynamic");
                    }
                })
                // Проверяем, что таблица имеет правильную схему
                .thenCompose(ignored -> getTableSchema(path))
                .thenAccept(currentSchema -> {
                    if (!schema.equals(currentSchema)) {
                        throw new IllegalStateException(
                                "Table " + path + " has a different schema (want: " + schema + ", have: "
                                        + currentSchema
                                        + ")");
                    }
                })
                // Проверяем ключи шардирования и монтируем таблицу
                .thenCompose(ignored -> maybeReshardAndMount(path, pivotKeys));
    }

    @Override
    public <T> CompletableFuture<T> runTransaction(Function<? super Transaction, ? extends CompletableFuture<T>> fn) {
        return YtSupportUtil.runBasicTransaction(startTransaction(), fn, executor());
    }
}
