package ru.yandex.direct.mysql.ytsync.common.keys;

import java.math.BigInteger;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Objects;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTree;
import ru.yandex.inside.yt.kosher.ytree.YTreeListNode;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;
import ru.yandex.misc.lang.number.UnsignedLong;

public class PivotKeys {
    private static final BigInteger FULL_PIVOT_HASH_RANGE = BigInteger.valueOf(1).shiftLeft(64);

    private final List<YTreeListNode> keys;

    public PivotKeys(List<YTreeListNode> keys) {
        this.keys = Objects.requireNonNull(keys);
        if (keys.isEmpty()) {
            throw new IllegalArgumentException("There must be at least one pivot key");
        }
        if (!keys.get(0).asList().isEmpty()) {
            throw new IllegalArgumentException("First pivot key must be empty");
        }
    }

    public List<YTreeListNode> getKeys() {
        return Collections.unmodifiableList(keys);
    }

    public int size() {
        return keys.size();
    }

    public static PivotKeys singlePartition() {
        return new PivotKeys(Collections.singletonList(YTree.builder().beginList().buildList()));
    }

    public static PivotKeys unsignedModulusPartitions(int modulus, int partitions) {
        List<YTreeListNode> keys = new ArrayList<>();
        keys.add(YTree.builder().beginList().buildList());
        int step = (modulus + partitions - 1) / partitions;
        int next = step;
        for (int i = 1; i < partitions; ++i) {
            keys.add(YTree.builder().beginList().value(UnsignedLong.valueOf(next)).buildList());
            next += step;
        }
        return new PivotKeys(keys);
    }

    public static PivotKeys unsignedHashPartitions(int partitions) {
        List<YTreeListNode> keys = new ArrayList<>();
        keys.add(YTree.builder().beginList().buildList());
        long step = FULL_PIVOT_HASH_RANGE.divide(BigInteger.valueOf(partitions)).longValue();
        long next = step;
        for (int i = 1; i < partitions; ++i) {
            keys.add(YTree.builder().beginList().value(UnsignedLong.valueOf(next)).buildList());
            next += step;
        }
        return new PivotKeys(keys);
    }

    public static PivotKeys signedHashPartitions(int partitions) {
        List<YTreeListNode> keys = new ArrayList<>();
        keys.add(YTree.builder().beginList().buildList());
        long step = FULL_PIVOT_HASH_RANGE.divide(BigInteger.valueOf(partitions)).longValue();
        long next = Long.MIN_VALUE + step;
        for (int i = 1; i < partitions; ++i) {
            keys.add(YTree.builder().beginList().value(next).buildList());
            next += step;
        }
        return new PivotKeys(keys);
    }

    /**
     * Генерирует PivotKeys для farm_hash % partitions
     * Например для partitions = 3 ответ будет [[], [1], [2]]
     */
    public static PivotKeys hashModPartitions(int partitions) {
        List<YTreeListNode> keys = new ArrayList<>();
        keys.add(YTree.builder().beginList().buildList());
        for (int i = 1; i < partitions; ++i) {
            keys.add(YTree.builder().beginList().value(i).buildList());
        }
        return new PivotKeys(keys);
    }

    public static PivotKeys unsignedPivots(long... values) {
        List<YTreeListNode> keys = new ArrayList<>();
        keys.add(YTree.builder().beginList().buildList());
        for (long value : values) {
            keys.add(YTree.builder().beginList().value(UnsignedLong.valueOf(value)).buildList());
        }
        return new PivotKeys(keys);
    }

    public static PivotKeys signedPivots(long... values) {
        List<YTreeListNode> keys = new ArrayList<>();
        keys.add(YTree.builder().beginList().buildList());
        for (long value : values) {
            keys.add(YTree.builder().beginList().value(value).buildList());
        }
        return new PivotKeys(keys);
    }

    public List<List<YTreeNode>> toKosherReshardTableKeys() {
        List<List<YTreeNode>> kosherKeys = Cf.arrayList();
        for (YTreeListNode key : keys) {
            ListF<YTreeNode> kosherKey = Cf.arrayList();
            for (YTreeNode node : key.asList()) {
                kosherKey.add(node);
            }
            kosherKeys.add(kosherKey);
        }
        return kosherKeys;
    }

    @Override
    public String toString() {
        return "PivotKeys{" +
                "keys=" + keys +
                '}';
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof PivotKeys)) {
            return false;
        }

        PivotKeys pivotKeys = (PivotKeys) o;

        return keys.equals(pivotKeys.keys);
    }

    @Override
    public int hashCode() {
        return keys.hashCode();
    }
}
