package ru.yandex.direct.mysql.ytsync.common.row;

import java.util.Arrays;
import java.util.List;
import java.util.Objects;

import ru.yandex.inside.yt.kosher.ytree.YTreeNode;

/**
 * Абстракция над плоским массивом колонок без определённой схемы
 * <p>
 * Значение колонки считается неизвестным, если оно равно null
 */
public final class FlatRow extends FlatRowView {
    private final YTreeNode[] columns;

    /**
     * Конструирует пустую строку с length колонками
     */
    public FlatRow(int length) {
        this.columns = new YTreeNode[length];
    }

    private FlatRow(YTreeNode[] columns) {
        this.columns = Objects.requireNonNull(columns);
    }

    @Override
    public YTreeNode get(int index) {
        return columns[index];
    }

    @Override
    public YTreeNode set(int index, YTreeNode element) {
        YTreeNode old = columns[index];
        columns[index] = element;
        return old;
    }

    @Override
    public int size() {
        return columns.length;
    }

    /**
     * Возвращает true, если в строке есть отсутствующие (null) колонки
     */
    public boolean hasMissingValues() {
        for (int i = 0; i < columns.length; i++) {
            if (columns[i] == null) {
                return true;
            }
        }
        return false;
    }

    /**
     * Возвращает true, если в строке есть присутствующие (не null) колонки
     */
    public boolean hasPresentValues() {
        for (int i = 0; i < columns.length; i++) {
            if (columns[i] != null) {
                return true;
            }
        }
        return false;
    }

    /**
     * Заменяет все отсутствующие (null) колонки на value
     */
    public void fillMissing(YTreeNode value) {
        for (int i = 0; i < columns.length; i++) {
            if (columns[i] == null) {
                columns[i] = value;
            }
        }
    }

    /**
     * Применяет обновлённые (не null) колонки из updates
     */
    public void update(List<YTreeNode> updates) {
        for (int i = 0; i < columns.length && i < updates.size(); i++) {
            YTreeNode newValue = updates.get(i);
            if (newValue != null) {
                columns[i] = newValue;
            }
        }
    }

    /**
     * Применяет обновлённые (не null) колонки из updates, но только если текущее значение равно null
     */
    public void updateWeakly(List<YTreeNode> updates) {
        for (int i = 0; i < columns.length && i < updates.size(); i++) {
            if (columns[i] == null) {
                YTreeNode newValue = updates.get(i);
                if (newValue != null) {
                    columns[i] = newValue;
                }
            }
        }
    }

    /**
     * Возвращает копию первых newLength колонок строки
     */
    public FlatRow first(int newLength) {
        if (newLength > columns.length) {
            throw new IllegalArgumentException("newLength cannot be greater than the row size");
        } else if (newLength < 0) {
            throw new IllegalArgumentException("newLength cannot be less than zero");
        }
        return new FlatRow(Arrays.copyOf(columns, newLength, YTreeNode[].class));
    }

    /**
     * Возвращает копию строки расширенной до newLength колонок
     */
    public FlatRow extend(int newLength) {
        if (newLength < columns.length) {
            throw new IllegalArgumentException("newLength cannot be smaller than the row size");
        }
        return new FlatRow(Arrays.copyOf(columns, newLength, YTreeNode[].class));
    }

    /**
     * Возвращает копию всех колонок
     */
    public FlatRow copy() {
        return new FlatRow(Arrays.copyOf(columns, columns.length, YTreeNode[].class));
    }

    /**
     * Возвращает копию колонок начиная from и заканчивая (но не включая) to
     */
    public FlatRow copy(int from, int to) {
        return new FlatRow(Arrays.copyOfRange(columns, from, to, YTreeNode[].class));
    }

    /**
     * Конструирует строку с одной колонкой и значением value
     */
    public static FlatRow of(YTreeNode value) {
        FlatRow row = new FlatRow(1);
        row.columns[0] = value;
        return row;
    }

    /**
     * Конструирует строку с указанными значениями колонок
     */
    public static FlatRow of(YTreeNode... values) {
        FlatRow row = new FlatRow(values.length);
        System.arraycopy(values, 0, row.columns, 0, values.length);
        return row;
    }

    /**
     * Конструирует строку с указанными значениями колонок
     */
    public static FlatRow of(List<YTreeNode> values) {
        FlatRow row = new FlatRow(values.size());
        int index = 0;
        for (YTreeNode value : values) {
            row.columns[index++] = value;
        }
        return row;
    }

    @Override
    public String toString() {
        return "FlatRow{columns=" + Arrays.toString(columns) + "}";
    }
}
