package ru.yandex.direct.mysql.ytsync.common.row;

import java.io.IOException;
import java.math.BigDecimal;
import java.util.function.Function;
import java.util.function.Predicate;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.annotations.VisibleForTesting;

import ru.yandex.direct.mysql.ytsync.common.model.JsonString;
import ru.yandex.inside.yt.kosher.impl.YtUtils;
import ru.yandex.inside.yt.kosher.impl.ytree.YTreeBooleanNodeImpl;
import ru.yandex.inside.yt.kosher.impl.ytree.YTreeDoubleNodeImpl;
import ru.yandex.inside.yt.kosher.impl.ytree.YTreeIntegerNodeImpl;
import ru.yandex.inside.yt.kosher.impl.ytree.YTreeStringNodeImpl;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTree;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;
import ru.yandex.misc.lang.number.UnsignedInteger;
import ru.yandex.misc.lang.number.UnsignedLong;

import static ru.yandex.direct.utils.CommonUtils.nvl;

public class FlatRowFieldProcessor<T, D> {
    public static final YTreeNode YTREE_EMPTY_NODE = YTree.builder().entity().build();
    private final String name;
    private final Function<String, YTreeNode> defaultValueProvider;
    private final Function<D, T> valueExtractor;
    private final Predicate<D> needDefault;
    private final Function<T, ?> transformer;

    public FlatRowFieldProcessor(String name,
                                 Function<String, YTreeNode> defaultValueProvider, Function<D, T> valueExtractor,
                                 Predicate<D> needDefault, Function<T, ?> transformer) {
        this.name = name;
        this.defaultValueProvider = defaultValueProvider;
        this.valueExtractor = valueExtractor;
        this.needDefault = needDefault;
        this.transformer = nvl(transformer, Function.identity());
    }

    public YTreeNode createNode(String dbName, D sourceData) {
        if (needDefault.test(sourceData)) {
            if (defaultValueProvider == null) {
                throw new IllegalStateException("Empty value provider for index = -1");
            }
            return defaultValueProvider.apply(dbName);
        }

        T value = valueExtractor.apply(sourceData);
        Object patchedValue = transformer.apply(value);
        if (patchedValue == null) {
            if (defaultValueProvider == null) {
                return YTREE_EMPTY_NODE;
            } else {
                return defaultValueProvider.apply(dbName);
            }
        } else {
            return buildNode(patchedValue);
        }
    }

    @VisibleForTesting
    static YTreeNode buildNode(Object value) {
        // частичный инлайнинг YTree.builder().value(value).build();
        // с фолбеком на оригинал
        if (value == null) {
            return YTREE_EMPTY_NODE;

        } else if (value instanceof JsonString) {
            ObjectMapper mapper = new ObjectMapper();
            try {
                JsonNode node = mapper.readTree(((JsonString) value).getJsonString().getBytes());
                return YtUtils.json2yson(YTree.builder(), node).build();
            } catch (RuntimeException | IOException e) {
                throw new IllegalStateException("got an error while converting json to yson", e);
            }
        } else if (value instanceof String) {
            return new YTreeStringNodeImpl((String) value, null);
        } else if (value instanceof byte[]) {
            return new YTreeStringNodeImpl((byte[]) value, null);

        } else if (value instanceof Integer) {
            return new YTreeIntegerNodeImpl(true, ((Integer) value).longValue(), null);
        } else if (value instanceof UnsignedInteger) {
            return new YTreeIntegerNodeImpl(false, ((UnsignedInteger) value).longValue(), null);
        } else if (value instanceof Long) {
            return new YTreeIntegerNodeImpl(true, (Long) value, null);
        } else if (value instanceof UnsignedLong) {
            return new YTreeIntegerNodeImpl(false, ((UnsignedLong) value).longValue(), null);

        } else if (value instanceof Boolean) {
            return new YTreeBooleanNodeImpl((Boolean) value, null);

        } else if (value instanceof Float) {
            return new YTreeDoubleNodeImpl((Float) value, null);
        } else if (value instanceof Double) {
            return new YTreeDoubleNodeImpl((Double) value, null);
        } else if (value instanceof BigDecimal) {
            return new YTreeDoubleNodeImpl(((BigDecimal) value).doubleValue(), null);

        } else {
            return YTree.builder().value(value).build();
        }
    }

    public String getName() {
        return name;
    }
}
