package ru.yandex.direct.mysql.ytsync.common.util;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Consumer;

import static java.util.Collections.emptyList;

/**
 * Позволяет распараллелить задачу в несколько потоков
 * <p>
 * Останавливает все задачи, если какая-либо из подзадач завершается с ошибкой
 */
public class ParallelRunner<T> {
    private static final AtomicInteger RUN_INDEX = new AtomicInteger();

    private final List<T> values;
    private final Consumer<T> consumer;
    private final Waiter waiter = new Waiter();
    private final AtomicBoolean runFlag = new AtomicBoolean(false);
    private volatile List<Thread> threads = emptyList();

    public ParallelRunner(List<T> values, Consumer<T> consumer) {
        this.values = values;
        this.consumer = consumer;
    }

    /**
     * Выделено в отдельный метод, чтобы sonar не ругался на try/catch
     */
    private static <T> void processValue(Waiter waiter, Consumer<T> consumer, T value) {
        try {
            consumer.accept(value);
        } catch (Exception e) {
            waiter.finish(e);
            return;
        }
        waiter.finish();
    }

    /**
     * Параллельно выполняет worker с каждым значением из values
     */
    public void run() throws InterruptedException {
        if (runFlag.getAndSet(true)) {
            throw new IllegalStateException("Parallel runner already started");
        }
        int runIndex = RUN_INDEX.incrementAndGet();
        int workerCount = 0;
        List<Thread> threads = new ArrayList<>();
        for (final T value : values) {
            waiter.add();
            threads.add(new Thread(
                    () -> processValue(waiter, consumer, value),
                    "parallel-run-" + runIndex + "-worker-" + (++workerCount)));
        }
        this.threads = threads;
        if (threads.isEmpty()) {
            return;
        }
        try (ThreadsJoiner ignored = new ThreadsJoiner(threads)) {
            for (Thread thread : threads) {
                thread.start();
            }
            waiter.await();
        }
    }

    public void stop() {
        waiter.finish(new RuntimeException("Parallel runner stopped from outside"));
    }

    /**
     * Примитив, который позволяет ожидать дочерние потоки до первой ошибки
     */
    private static class Waiter {
        private int count;
        private Throwable failure;

        public synchronized void add() {
            ++count;
        }

        synchronized void await() throws InterruptedException {
            // yukaba: наблюдал ситуацию, когда поток падал с "java.lang.OutOfMemoryError: Java heap space", который не ловился, count для потока не уменьшался и цикл продолжался бесконечно
            while (count > 0 && failure == null) {
                wait();
            }
            if (failure != null) {
                if (failure instanceof RuntimeException) {
                    throw (RuntimeException) failure;
                } else if (failure instanceof Error) {
                    throw (Error) failure;
                } else {
                    throw new IllegalStateException("Unexpected exception", failure);
                }
            }
        }

        synchronized void finish(Throwable failure) {
            if (count > 0) {
                --count;
                if (failure != null && this.failure == null) {
                    // Запоминаем первое исключение
                    this.failure = failure;
                    notifyAll();
                } else if (count == 0) {
                    // Все потоки завершились
                    notifyAll();
                }
            }
        }

        void finish() {
            finish(null);
        }
    }

    public int getAliveThreadsCount() {
        List<Thread> threads = this.threads;
        return (int) threads.stream().filter(Thread::isAlive).count();
    }

    /**
     * Автоматически interrupt'ит и дожидается завершения всех потоков по списку
     */
    private static class ThreadsJoiner implements AutoCloseable {
        private final List<Thread> threads;

        ThreadsJoiner(List<Thread> threads) {
            this.threads = threads;
        }

        @Override
        public void close() {
            for (Thread thread : threads) {
                thread.interrupt();
            }
            boolean interrupted = false;
            for (Thread thread : threads) {
                while (thread.isAlive()) {
                    try {
                        thread.join(); // IS-NOT-COMPLETABLE-FUTURE-JOIN
                    } catch (InterruptedException e) {
                        interrupted = true;
                    }
                }
            }
            if (interrupted) {
                Thread.currentThread().interrupt();
            }
        }
    }
}
