package ru.yandex.direct.mysql.ytsync.common.util;

import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.ObjLongConsumer;
import java.util.function.Supplier;

import com.google.common.base.Preconditions;
import com.google.common.util.concurrent.UncheckedExecutionException;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.mysql.ytsync.common.compatibility.YtSupport;
import ru.yandex.direct.utils.InterruptedRuntimeException;
import ru.yandex.direct.utils.ThreadUtils;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.yt.ytclient.rpc.RpcUtil;
import ru.yandex.yt.ytclient.tables.TableSchema;

public class YtSyncCommonUtil {
    public static final Logger YTSYNC_LOGGER = LoggerFactory.getLogger("YTSYNC.log");

    private static final Logger logger = LoggerFactory.getLogger(YtSyncCommonUtil.class);

    private YtSyncCommonUtil() {
    }

    private static <T> T execFuncWithRetries(Function<Integer, T> func) {
        return ThreadUtils.execFuncWithRetries(func, 10, 1000, 1.4, logger);
    }

    private static void execWithRetries(Consumer<Integer> action) {
        ThreadUtils.execWithRetries(action, 10, 1000, 1.4, logger);
    }

    /**
     * Проверяет существование динамической таблицы path со схемой schema
     */
    public static void verifyDynamicTable(YtSupport ytSupport, String path, TableSchema schema) {
        logger.info("Verifying schema of {}", path);
        Pair<TableSchema, Boolean> schemaAndDynamic = execFuncWithRetries(
                attempt -> ytSupport.getTableSchemaAndIsDynamic(path).join() // IGNORE-BAD-JOIN DIRECT-149116
        );
        Boolean dynamic = schemaAndDynamic.getRight();
        TableSchema currentSchema = schemaAndDynamic.getLeft();
        if (!dynamic) {
            throw new IllegalStateException("Table " + path + " is not dynamic");
        }
        if (!schema.equals(currentSchema)) {
            throw new IllegalStateException(
                    "Table " + path + " has a different schema (want: " + schema + ", have: " + currentSchema + ")");
        }
    }

    private static String getTabletState(YtSupport ytSupport, YPath table) {
        try {
            return ytSupport.getTabletState(table.toString()).get();
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new InterruptedRuntimeException(e);
        } catch (ExecutionException e) {
            throw new UncheckedExecutionException(e);
        }
    }

    /**
     * Проверяет состояние динамической таблицы и приводит её к состоянию mounted
     * Если unfreeze = true, то при текущем статусе таблицы frozen будет выполнен unfreeze
     */
    public static void makeTableMounted(YtSupport ytSupport, String path, boolean unfreeze) {
        String tabletState = execFuncWithRetries(attempt ->
                ytSupport.getTabletState(path).join()); // IGNORE-BAD-JOIN DIRECT-149116
        if (!"mounted".equals(tabletState)) {
            if ("unmounted".equals(tabletState)) {
                logger.info("Mounting {}", path);
                execWithRetries(attempt -> {
                    if (attempt > 1) {
                        // Возможно, mount уже сработал, и повторный запрос не требуется
                        String state = getTabletState(ytSupport, YPath.simple(path));
                        if (state.equals("mounted")) {
                            return;
                        }
                    }
                    ytSupport.mountTable(path).join(); // IGNORE-BAD-JOIN DIRECT-149116
                });
                execWithRetries(attempt -> ytSupport.waitTableMounted(path).join()); // IGNORE-BAD-JOIN DIRECT-149116
            } else if ("frozen".equals(tabletState)) {
                if (unfreeze) {
                    logger.info("Unfreezing {}", path);
                    execWithRetries(attempt -> {
                        if (attempt > 1) {
                            // Возможно, анфриз уже сработал, и повторный запрос не требуется
                            String state = getTabletState(ytSupport, YPath.simple(path));
                            if (state.equals("mounted")) {
                                return;
                            }
                        }
                        ytSupport.unfreezeTable(path).join(); // IGNORE-BAD-JOIN DIRECT-149116
                    });
                    execWithRetries(attempt -> ytSupport.waitTableMounted(path)
                            .join()); // IGNORE-BAD-JOIN DIRECT-149116
                }
            } else {
                // Может быть ещё состояние transient, но не понятно, что с ним делать
                // Наверное, можно подождать, пока оно сойдётся к какому-то определённому состоянию,
                // но пока проще выполнить рестарт
                throw new RuntimeException(String.format("Unexpected tablet_state \"%s\"", tabletState));
            }
        }
    }

    public static int extractShard(String dbName) {
        return ObjectUtils.firstNonNull(parseDbName(dbName).getShard(), 0);
    }

    public static DbShardPair parseDbName(String dbName) {
        if (!dbName.contains(":")) {
            return new DbShardPair(dbName, null);
        }
        String[] split = dbName.split(":");
        return new DbShardPair(split[0], Integer.parseInt(split[1]));
    }

    public static class DbShardPair {
        private final String db;
        private final Integer shard;

        public DbShardPair(String db, Integer shard) {
            this.db = db;
            this.shard = shard;
        }

        public String getDb() {
            return db;
        }

        public Integer getShard() {
            return shard;
        }
    }

    public static <T> CompletableFuture<Void> allFutures(List<CompletableFuture<T>> futures) {
        return CompletableFuture.allOf(futures.toArray(new CompletableFuture[0]));
    }

    /**
     * Выполняет операцию с замером времени выполнения в наносекундах
     */
    public static <T> CompletableFuture<T> startMeasured(Runnable before, Supplier<CompletableFuture<T>> start,
                                                         ObjLongConsumer<T> after) {
        before.run();
        long t0 = System.nanoTime();
        return RpcUtil.apply(start.get(), result -> {
            long t1 = System.nanoTime();
            after.accept(result, t1 - t0);
            return result;
        });
    }

    /**
     * Возвращает название директории вида "ppc:1-ppc:2-ppc:3"
     * Используется для заведения локов и хранения информации, нужной конкретному инстансу,
     * который обслуживает базы данных dbNames
     */
    public static String getAllTablesImportInitialSubDir(List<String> dbNames) {
        return String.join("-", dbNames);
    }

    /**
     * Ожидание готовности всех futures
     */
    public static void waitAll(List<Future<?>> futures) {
        Preconditions.checkNotNull(futures);
        //
        for (Future<?> future : futures) {
            try {
                future.get();
            } catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                throw new RuntimeException(e);
            } catch (ExecutionException e) {
                throw new RuntimeException(e);
            }
        }
    }
}
