package ru.yandex.direct.mysql.ytsync.synchronizator.monitoring;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Timer;
import java.util.TimerTask;
import java.util.function.Consumer;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Component;

import ru.yandex.direct.mysql.ytsync.common.compatibility.YtSupport;
import ru.yandex.direct.mysql.ytsync.synchronizator.util.SyncConfig;
import ru.yandex.direct.mysql.ytsync.synchronizator.util.YtSyncUtil;
import ru.yandex.direct.ytwrapper.YtPathUtil;

/**
 * Определяет состояние синхронизатора. Алгоритм:
 * По ссылке определяет папку из которой grid-core читает данные
 * Сравнивает путь этой папки с путем куда записывает данные синхронизатор
 * Если пути совпадают, то значит {@link SyncState#STABLE} иначе {@link SyncState#PRESTABLE}
 * <p>
 * Значение состояния обновляется каждые {@link #RELOAD_INTERVAL}
 */
@Lazy
@Component
@ParametersAreNonnullByDefault
public class SyncStateChecker {
    private static final Logger logger = LoggerFactory.getLogger(SyncStateChecker.class);

    private static final Duration RELOAD_INTERVAL = Duration.ofMinutes(1);

    private final String stableLinkPath;
    private final String rootPath;
    private final YtSupport ytSupport;

    private volatile SyncState syncState;

    private final List<Consumer<SyncState>> subscribers = Collections.synchronizedList(new ArrayList<>());

    @Autowired
    public SyncStateChecker(SyncConfig syncConfig, YtSupport ytSupport) {
        this.stableLinkPath = YtPathUtil.normalizePath(syncConfig.stableLinkPath());
        this.rootPath = YtPathUtil.normalizePath(syncConfig.rootPath());
        this.ytSupport = ytSupport;
        this.syncState = checkSyncState();
        logger.info("Init syncState = {}", this.syncState);
        //
        new Timer("sync-state-refresh-timer", true)
                .schedule(new TimerTask() {
                    @Override
                    public void run() {
                        try {
                            SyncState nextSyncState = checkSyncState();
                            if (syncState != nextSyncState) {
                                syncState = nextSyncState;
                                logger.info("Changed syncState to {}", nextSyncState);
                                //
                                List<Consumer<SyncState>> subscribersCopy;
                                synchronized (subscribers) {
                                    subscribersCopy = new ArrayList<>(subscribers);
                                }
                                for (Consumer<SyncState> handler : subscribersCopy) {
                                    handler.accept(nextSyncState);
                                }
                            }
                        } catch (Exception e) {
                            logger.error("Unhandled exception in timer func, ignoring", e);
                        }
                    }
                }, RELOAD_INTERVAL.toMillis(), RELOAD_INTERVAL.toMillis());
    }

    /**
     * Определяет состояние синхронизатора
     * Если не удалось получить путь на который указывает ссылка, то возвращает {@link SyncState#PRESTABLE}
     */
    private SyncState checkSyncState() {
        String targetPath = YtSyncUtil.readLinkTargetPath(ytSupport, stableLinkPath);
        if (targetPath != null) {
            targetPath = YtPathUtil.normalizePath(targetPath);
        }
        return rootPath.equals(targetPath) ? SyncState.STABLE : SyncState.PRESTABLE;
    }

    public SyncState getSyncState() {
        return syncState;
    }

    public void subscribe(Consumer<SyncState> handler) {
        subscribers.add(handler);
    }

    public void unsubscribe(Consumer<SyncState> handler) {
        subscribers.remove(handler);
    }
}
