package ru.yandex.direct.mysql.ytsync.synchronizator.snapshot;

import java.time.Duration;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.function.BiConsumer;
import java.util.function.Consumer;

import com.google.common.base.Preconditions;
import com.google.common.util.concurrent.UncheckedExecutionException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.direct.mysql.ytsync.common.compatibility.YtSupport;
import ru.yandex.direct.mysql.ytsync.common.components.SyncStatesConfig;
import ru.yandex.direct.mysql.ytsync.common.components.SyncStatesTable;
import ru.yandex.direct.mysql.ytsync.common.util.YtSyncCommonUtil;
import ru.yandex.direct.mysql.ytsync.synchronizator.monitoring.SyncState;
import ru.yandex.direct.mysql.ytsync.synchronizator.monitoring.SyncStateChecker;
import ru.yandex.direct.mysql.ytsync.synchronizator.util.SnapshotConfig;
import ru.yandex.direct.mysql.ytsync.synchronizator.util.SyncConfig;
import ru.yandex.direct.utils.InterruptedRuntimeException;
import ru.yandex.direct.utils.ThreadUtils;
import ru.yandex.direct.ytwrapper.YtUtils;
import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;
import ru.yandex.inside.yt.kosher.ytree.YTreeStringNode;

import static ru.yandex.direct.mysql.ytsync.common.util.YtSyncCommonUtil.waitAll;
import static ru.yandex.direct.ytwrapper.YtUtils.getRecursiveUsageByMediums;
import static ru.yandex.direct.ytwrapper.YtUtils.getRemainingSpaceByMediums;

/**
 * Код, умеющий делать снепшоты таблиц зеркализатора.
 * Каждый инстанс копирует свои данные в общую директорию вида
 * //home/direct/mysql-sync/snapshot--v.14--2019-12-19/
 * Таблица mysql-sync-states общая, поэтому она не копируется целиком, а создаётся изначально пустая,
 * а строчки в ней дописываются инстансами по мере обработки их dbNames.
 * В результате получается структура, идентичная исходной.
 */
public class SnapshotService {
    private static final Logger logger = LoggerFactory.getLogger(SnapshotService.class);

    // Атрибут на директориях вида "ppc:1"
    public static final String SNAPSHOT_FINISHED = "snapshot-finished";
    // Название снепшота, например "snapshot--v.14--2019-12-19"
    public static final String SNAPSHOT_NAME_TEMPLATE = "snapshot--%s--%s";
    // TTL снепшота, на базе него будет установлен в атрибут @expiration_time
    public static final Duration SNAPSHOT_TTL = Duration.ofHours(23);

    private final SyncConfig syncConfig;
    private final SyncStatesConfig syncStatesConfig;
    private final Yt yt;
    private final YtSupport ytSupport;
    private final SnapshotConfig snapshotConfig;
    private final SyncStateChecker syncStateChecker;

    // Дата успешного снятия снепшота (не обязательно этим инстансом)
    // Чтобы не проверять детально наличие снепшота за сегодня
    private LocalDate successSnapshotDate = null;

    public SnapshotService(SyncConfig syncConfig, SyncStatesConfig syncStatesConfig, Yt yt, YtSupport ytSupport,
                           SyncStateChecker syncStateChecker) {
        this.syncConfig = syncConfig;
        this.syncStatesConfig = syncStatesConfig;
        this.yt = yt;
        this.ytSupport = ytSupport;
        this.syncStateChecker = syncStateChecker;
        //
        this.snapshotConfig = syncConfig.snapshotConfig();
        logger.info("Loaded snapshotConfig={}", snapshotConfig);
    }

    public boolean needSnapshotNow() {
        if (!syncConfig.importAllTables()) {
            // Для гридового репликатора не делаем снепшотов
            return false;
        }
        if (!snapshotConfig.isEnabled()) {
            return false;
        }
        LocalDateTime now = LocalDateTime.now();
        LocalTime time = now.toLocalTime();
        LocalDate date = now.toLocalDate();
        if (!time.isAfter(snapshotConfig.getIntervalFrom()) || !time.isBefore(snapshotConfig.getIntervalTo())) {
            return false;
        }
        // Проверить, не сделан ли уже снепшот за сегодня
        if (date.equals(successSnapshotDate)) {
            return false;
        }
        // Проверить, не престейбл ли этот инстанс
        if (syncStateChecker.getSyncState() == SyncState.PRESTABLE && !snapshotConfig.isEnabledOnPrestable()) {
            logger.info("Not enabled on prestable");
            return false;
        }
        // Если не известно, проверить более детально (в Кипарисе)
        YPath snapshotPath = getSnapshotPath();
        if (yt.cypress().exists(snapshotPath)) {
            boolean snapshotFinished = true;
            for (String dbName : syncConfig.getDbNames()) {
                if (!yt.cypress().exists(snapshotPath.child(dbName))) {
                    snapshotFinished = false;
                    break;
                }
                YTreeNode node = yt.cypress().get(snapshotPath.child(dbName), Cf.set(SNAPSHOT_FINISHED));
                if (!node.getAttribute(SNAPSHOT_FINISHED).isPresent()) {
                    snapshotFinished = false;
                    break;
                }
                if (!node.getAttributeOrThrow(SNAPSHOT_FINISHED).boolValue()) {
                    snapshotFinished = false;
                    break;
                }
            }
            if (snapshotFinished) {
                logger.info("Found success snapshot for today: {}", snapshotPath);
                successSnapshotDate = date;
                return false;
            }
        }
        // Проверяем, есть ли свободное место для снепшота
        if (!isEnoughSpace()) {
            return false;
        }
        return true;
    }

    /**
     * Возвращает true, если на аккаунте YT есть достаточно свободного места (1.5 * текущий размер)
     * false в противном случае
     */
    private boolean isEnoughSpace() {
        String account = yt.cypress().get(YPath.simple(syncConfig.rootPath()), Cf.set("account"))
                .getAttributeOrThrow("account").stringValue();
        YPath basedir = YPath.simple(syncConfig.rootPath());
        String medium = syncStatesConfig.syncTableMedium();
        Map<String, Long> recursiveUsageByMediums = getRecursiveUsageByMediums(yt, basedir);
        Map<String, Long> remainingSpaceByMediums = getRemainingSpaceByMediums(yt, account);
        if (!remainingSpaceByMediums.containsKey(medium)) {
            logger.error("Can't get remaining space on medium = {}", medium);
            return false;
        }
        long used = recursiveUsageByMediums.getOrDefault(medium, 0L);
        long remaining = remainingSpaceByMediums.get(medium);
        if (remaining < used * 3 / 2) {
            logger.error("Remaining space is {} and it's not enough to safe snapshot (need 1.5 * {} bytes)",
                    remaining, used);
            return false;
        }
        logger.info("Remaining space: {} bytes ({} GB), it's OK", remaining, remaining / (1024 * 1024 * 1024));
        return true;
    }

    private YPath getSnapshotPath() {
        String rootPath = syncConfig.rootPath();
        String version = YPath.simple(rootPath).name();
        String name = String.format(SNAPSHOT_NAME_TEMPLATE, version,
                LocalDate.now().format(DateTimeFormatter.ISO_LOCAL_DATE));
        return YPath.simple(rootPath).parent().child(name);
    }

    public void createSnapshot() {
        Preconditions.checkState(syncConfig.importAllTables());

        YPath rootPath = YPath.simple(syncConfig.rootPath());
        YPath rootSnapshotPath = getSnapshotPath();

        // Будем копировать те строчки mysql-sync-states, которые относятся к обрабатываемым dbNames
        SyncStatesTable srcSyncStatesTable = new SyncStatesTable(
                ytSupport,
                syncStatesConfig.syncStatesPath(),
                syncStatesConfig.syncTableMedium(),
                ignored -> {
                });

        SyncStatesTable dstSyncStatesTable = new SyncStatesTable(
                ytSupport,
                rootSnapshotPath.child(YPath.simple(srcSyncStatesTable.getPath()).name()).toString(),
                syncStatesConfig.syncTableMedium(),
                ignored -> {
                });
        execWithRetries(attempt -> dstSyncStatesTable.prepareTable());

        // Устанавливаем TTL для снепшота
        logger.info("Set TTL={} for snapshot directory {}", SNAPSHOT_TTL, rootSnapshotPath);
        execWithRetries(attempt -> yt.cypress().set(
                rootSnapshotPath.attribute(YtUtils.EXPIRATION_TIME_ATTR),
                Duration.ofMillis(new Date().getTime()).plus(SNAPSHOT_TTL).toMillis()
        ));

        // Нужно заморозить и скопировать все директории вида "$rootPath/$dbName"
        List<String> dbNames = syncConfig.getDbNames();
        // Размер пула 10 чтобы наши инстансы не задалбывали YT запросами. Ограничение на одновременное
        // кол-во запросов сейчас -- 100 RPS, если будет больше, YT начнёт массово отвечать ошибками
        ExecutorService executorService = Executors.newFixedThreadPool(10);
        try {
            Set<YPath> copiedDirectories = new HashSet<>();
            for (String dbName : dbNames) {
                YPath dbSource = rootPath.child(dbName);
                //
                Set<YPath> tables = getTablesWithState(dbSource, "mounted");
                waitAll(beginFreezeTables(tables, executorService));
                waitAll(beginWaitTabletState(tables, "frozen", executorService));
                //
                YPath dbSnapshotPath = rootSnapshotPath.child(dbName);
                //
                logger.info("Copying all content from {} to {}", dbSource, dbSnapshotPath);
                execWithRetries(attempt -> yt.cypress().copy(dbSource, dbSnapshotPath, true, true, true));
                copiedDirectories.add(dbSource);
                //
                logger.info("Copying current state for {} from {} to {}",
                        dbName,
                        srcSyncStatesTable.getPath(),
                        dstSyncStatesTable.getPath());
                Optional<SyncStatesTable.SyncState> state = srcSyncStatesTable.lookupRow(dbName);
                dstSyncStatesTable.setState(dbName, state.get());
                ytSupport.runTransaction(dstSyncStatesTable::apply).join(); // IGNORE-BAD-JOIN DIRECT-149116
                dstSyncStatesTable.committed();
                //
                logger.info("Set {}=true on {}", SNAPSHOT_FINISHED, dbSnapshotPath);
                execWithRetries(attempt -> yt.cypress().set(dbSnapshotPath.attribute(SNAPSHOT_FINISHED), true));
            }

            // И ещё нужно скопировать директорию инстанса - там установлен атрибут initial-import-finished
            // Это нужно для удобства -- вдруг будет нужно запустить на снепшоте зеркализатор
            String instanceDirName = YtSyncCommonUtil.getAllTablesImportInitialSubDir(syncConfig.getDbNames());
            YPath srcInstanceDir = rootPath.child(instanceDirName);
            YPath dstInstanceDir = rootSnapshotPath.child(instanceDirName);
            // Если инстанс обслуживает один шард, директорией инстанса будет директория с данными
            // В этом случае снова копировать её не нужно -- она уже была скопирована кодом выше
            if (!copiedDirectories.contains(srcInstanceDir)) {
                logger.info("Copying instance dir {} to {}", srcInstanceDir, dstInstanceDir);
                execWithRetries(attempt -> yt.cypress().copy(srcInstanceDir, dstInstanceDir, true, true, true));
            }
        } finally {
            try {
                for (String dbName : dbNames) {
                    YPath dbSource = rootPath.child(dbName);
                    //
                    Set<YPath> tables = getTablesWithState(dbSource, "frozen");
                    waitAll(beginUnfreezeTables(tables, executorService));
                    waitAll(beginWaitTabletState(tables, "mounted", executorService));
                }
            } finally {
                // Без остановки этого пула программа не сможет завершиться
                logger.info("Stopping executorService..");
                executorService.shutdownNow();
                logger.info("Stopped executorService");
            }
        }
    }

    private Set<YPath> getTablesWithState(YPath dir, String state) {
        Set<YPath> tables = new HashSet<>();
        execWithRetries(attempt -> {
            tables.clear();
            forEachDynamicTable(dir, (table, tabletState) -> {
                if (state.equals(tabletState)) {
                    tables.add(table);
                }
            });
        });
        return tables;
    }

    private List<Future<?>> beginFreezeTables(Set<YPath> tables, ExecutorService executorService) {
        List<Future<?>> futures = new ArrayList<>();
        for (YPath table : tables) {
            futures.add(executorService.submit(() -> {
                logger.info("Freezing table {}", table);
                execWithRetries(attempt -> {
                    if (attempt > 1) {
                        // Возможно, фриз уже сработал, и повторный фриз не требуется
                        // нужно проверить статус таблицы
                        String tabletState = getTabletState(table);
                        if (tabletState.equals("frozen")) {
                            return;
                        }
                    }
                    yt.tables().freeze(table);
                });
            }));
        }
        return futures;
    }

    private String getTabletState(YPath table) {
        try {
            return ytSupport.getTabletState(table.toString()).get();
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new InterruptedRuntimeException(e);
        } catch (ExecutionException e) {
            throw new UncheckedExecutionException(e);
        }
    }

    private List<Future<?>> beginWaitTabletState(Set<YPath> tables, String state, ExecutorService executorService) {
        List<Future<?>> futures = new ArrayList<>();
        for (YPath table : tables) {
            futures.add(executorService.submit(() -> {
                execWithRetries(attempt -> {
                    ytSupport.waitTabletStates(table.toString(), state).join(); // IGNORE-BAD-JOIN DIRECT-149116
                });
            }));
        }
        return futures;
    }

    private List<Future<?>> beginUnfreezeTables(Set<YPath> tables, ExecutorService executorService) {
        List<Future<?>> futures = new ArrayList<>();
        for (YPath table : tables) {
            futures.add(executorService.submit(() -> {
                logger.info("Unfreezing table {}", table);
                execWithRetries(attempt -> {
                    if (attempt > 1) {
                        // Возможно, анфриз уже сработал, и повторный запрос не требуется
                        // нужно проверить статус таблицы
                        String tabletState = getTabletState(table);
                        if (tabletState.equals("mounted")) {
                            return;
                        }
                    }
                    yt.tables().unfreeze(table);
                });
            }));
        }
        return futures;
    }

    private void execWithRetries(Consumer<Integer> action) {
        ThreadUtils.execWithRetries(action, 10, 1000, 1.4, logger);
    }

    /**
     * Применяет action ко всем найденным в path динамическим таблицам (рекурсивно)
     */
    private void forEachDynamicTable(YPath path, BiConsumer<YPath, String> action) {
        List<YTreeStringNode> list = yt.cypress().list(path, Cf.set("type", "dynamic", "tablet_state"));
        for (YTreeStringNode node : list) {
            YTreeNode typeAttr = node.getAttributeOrThrow("type");
            if ("map_node".equals(typeAttr.stringValue())) {
                forEachDynamicTable(path.child(node.stringValue()), action);
            } else if ("table".equals(typeAttr.stringValue())) {
                if (node.getAttribute("dynamic").get().boolValue()) {
                    action.accept(
                            path.child(node.stringValue()),
                            node.getAttribute("tablet_state").get().stringValue());
                }
            }
        }
    }
}
