package ru.yandex.direct.mysql.ytsync.synchronizator.streamer;

import java.util.function.BooleanSupplier;

import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.PreDestroy;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.mysql.ytsync.common.compatibility.YtLockUtil;
import ru.yandex.direct.mysql.ytsync.common.compatibility.YtSupport;
import ru.yandex.direct.mysql.ytsync.common.components.SyncStatesConfig;
import ru.yandex.direct.mysql.ytsync.common.util.MySqlConnectionProvider;
import ru.yandex.direct.mysql.ytsync.common.util.YtSyncCommonUtil;
import ru.yandex.direct.mysql.ytsync.synchronizator.monitoring.SyncStateChecker;
import ru.yandex.direct.mysql.ytsync.synchronizator.snapshot.SnapshotService;
import ru.yandex.direct.mysql.ytsync.synchronizator.streamer.mysql.AggregatorLock;
import ru.yandex.direct.mysql.ytsync.synchronizator.streamer.mysql.MysqlTransactionStreamersPool;
import ru.yandex.direct.mysql.ytsync.synchronizator.streamer.yt.YtAggregatorLockWatcher;
import ru.yandex.direct.mysql.ytsync.synchronizator.streamer.yt.YtTransactionAggregatorFactory;
import ru.yandex.direct.mysql.ytsync.synchronizator.util.SyncConfig;
import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.inside.yt.kosher.cypress.CypressNodeType;
import ru.yandex.inside.yt.kosher.cypress.YPath;

@Component
@ParametersAreNonnullByDefault
public class MysqlToYtSyncStreamerCoordinator {
    private final SyncConfig syncConfig;
    private final Yt yt;
    private final YtSupport ytSupport;
    private final YtTransactionAggregatorFactory aggregatorFactory;
    private final MySqlConnectionProvider mySqlConnectionProvider;
    private final SyncStatesConfig syncStatesConfig;
    private final SyncStateChecker syncStateChecker;

    private MysqlTransactionStreamersPool coordinatedPool;

    @Autowired
    public MysqlToYtSyncStreamerCoordinator(SyncConfig syncConfig,
                                            SyncStatesConfig syncStatesConfig,
                                            YtTransactionAggregatorFactory aggregatorFactory,
                                            MySqlConnectionProvider mySqlConnectionProvider,
                                            Yt yt,
                                            YtSupport ytSupport,
                                            SyncStateChecker syncStateChecker) {
        this.syncConfig = syncConfig;
        this.aggregatorFactory = aggregatorFactory;
        this.mySqlConnectionProvider = mySqlConnectionProvider;
        this.yt = yt;
        this.ytSupport = ytSupport;
        this.syncStatesConfig = syncStatesConfig;
        this.syncStateChecker = syncStateChecker;
    }

    private YPath getSyncLockPath() {
        if (syncConfig.importAllTables()) {
            String dbNames = YtSyncCommonUtil.getAllTablesImportInitialSubDir(syncConfig.getDbNames());
            return YPath.simple(syncConfig.rootPath()).child(dbNames).child("sync-lock");
        } else {
            return YPath.simple(syncConfig.rootPath()).child("sync-lock");
        }
    }

    public void startSync() {
        if (coordinatedPool != null) {
            throw new IllegalStateException("Sync is already running");
        }

        // Для старта sync берём глобальный лок на instance
        YPath lockPath = getSyncLockPath();
        if (!yt.cypress().exists(lockPath)) {
            yt.cypress().create(lockPath, CypressNodeType.BOOLEAN, true, true);
        }
        BooleanSupplier interruptedSupplier = () -> Thread.currentThread().isInterrupted();
        YtLockUtil.runInLock(yt, lockPath, interruptedSupplier, transaction -> {
            if (interruptedSupplier.getAsBoolean()) {
                return;
            }
            // watcher будет следить за живостью лока и поинтерраптит тред, если лок будет потерян
            try (YtAggregatorLockWatcher lockWatcher =
                         new YtAggregatorLockWatcher(yt, transaction, Thread.currentThread())) {
                try {
                    startSyncInLock(lockWatcher);
                } catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                    throw new RuntimeException("Thread was interrupted", e);
                }
            }
        });
    }

    public void startSyncInLock(AggregatorLock lock) throws InterruptedException {
        aggregatorFactory.prepareSync();
        try (MysqlTransactionStreamersPool pool = new MysqlTransactionStreamersPool(syncConfig,
                aggregatorFactory, mySqlConnectionProvider, yt,
                new SnapshotService(syncConfig, syncStatesConfig, yt, ytSupport, syncStateChecker))) {
            coordinatedPool = pool;
            pool.syncInPool(lock);
        } finally {
            coordinatedPool = null;
        }
    }

    @PreDestroy
    public void close() {
        MysqlTransactionStreamersPool pool = coordinatedPool;
        if (pool != null) {
            coordinatedPool = null;
            pool.close();
        }
    }
}
