package ru.yandex.direct.mysql.ytsync.synchronizator.streamer.mysql;

import java.io.IOException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.sql.Connection;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.time.Instant;
import java.time.ZoneOffset;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.github.shyiko.mysql.binlog.GtidSet;
import com.google.common.base.Preconditions;
import com.google.common.base.Stopwatch;
import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.binlog.model.CreateTable;
import ru.yandex.direct.binlog.model.DropColumn;
import ru.yandex.direct.binlog.model.DropTable;
import ru.yandex.direct.binlog.model.Operation;
import ru.yandex.direct.binlog.model.RenameColumn;
import ru.yandex.direct.binlog.model.RenameTable;
import ru.yandex.direct.binlog.model.SchemaChange;
import ru.yandex.direct.binlogbroker.mysql.MysqlUtil;
import ru.yandex.direct.binlogbroker.replicatetoyt.AlterHandler;
import ru.yandex.direct.binlogbroker.replicatetoyt.DummyStateManager;
import ru.yandex.direct.mysql.BinlogDDLException;
import ru.yandex.direct.mysql.BinlogEvent;
import ru.yandex.direct.mysql.BinlogEventConverter;
import ru.yandex.direct.mysql.BinlogEventData;
import ru.yandex.direct.mysql.BinlogRawEventServerSource;
import ru.yandex.direct.mysql.BinlogRawEventSource;
import ru.yandex.direct.mysql.MySQLBinlogState;
import ru.yandex.direct.mysql.MySQLColumnData;
import ru.yandex.direct.mysql.MySQLServer;
import ru.yandex.direct.mysql.MySQLServerBuilder;
import ru.yandex.direct.mysql.MySQLSimpleRow;
import ru.yandex.direct.mysql.MySQLSimpleRows;
import ru.yandex.direct.mysql.ytsync.common.util.MySqlConnectionProvider;
import ru.yandex.direct.mysql.ytsync.common.util.MySqlConnectionSettings;
import ru.yandex.direct.mysql.ytsync.common.util.MySqlSettingsChangedListener;
import ru.yandex.direct.mysql.ytsync.synchronizator.operation.YtPendingDelete;
import ru.yandex.direct.mysql.ytsync.synchronizator.operation.YtPendingInsert;
import ru.yandex.direct.mysql.ytsync.synchronizator.operation.YtPendingOperation;
import ru.yandex.direct.mysql.ytsync.synchronizator.operation.YtPendingTransaction;
import ru.yandex.direct.mysql.ytsync.synchronizator.operation.YtPendingUpdate;
import ru.yandex.direct.mysql.ytsync.synchronizator.streamer.exceptions.FatalException;
import ru.yandex.direct.mysql.ytsync.synchronizator.util.SyncConfig;
import ru.yandex.direct.tracing.data.DirectTraceInfo;
import ru.yandex.direct.utils.InterruptedRuntimeException;
import ru.yandex.direct.utils.exception.RuntimeTimeoutException;
import ru.yandex.direct.utils.io.TempDirectory;
import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.inside.yt.kosher.cypress.YPath;

import static ru.yandex.direct.mysql.ytsync.synchronizator.streamer.mysql.ChecksumRecord.YT_CHECKSUM;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

public class MysqlTransactionStreamer extends Thread implements AutoCloseable {
    private static final Logger logger = LoggerFactory.getLogger(MysqlTransactionStreamer.class);

    private final MySqlConnectionProvider connectionProvider;
    private final String dbName;
    private final ServerIdSupplier serverIdSupplier;
    private final MysqlTransactionAggregator aggregator;
    private final CountDownLatch startupLatch;
    private final GtidSet stopAfter;

    private final MySqlSettingsChangedListener listener = this::mySqlSettingsChanged;
    private final AlterHandler alterHandler;
    private final boolean importAllTables;
    private final Set<String> excludeTableNames;
    private final SyncConfig syncConfig;
    private final boolean verifyChecksums;

    private volatile boolean settingsChanged = false;

    private Lock pauseLock = new ReentrantLock(true);
    private Condition pauseCond = pauseLock.newCondition();
    private volatile boolean pauseRequested = false;
    private volatile boolean paused = false;

    MysqlTransactionStreamer(MySqlConnectionProvider connectionProvider, String dbName,
                             ServerIdSupplier serverIdSupplier, MysqlTransactionAggregator aggregator,
                             CountDownLatch startupLatch,
                             GtidSet stopAfter, SyncConfig syncConfig, Yt yt) {
        super("binlog streamer for " + dbName);
        //
        this.connectionProvider = connectionProvider;
        this.dbName = dbName;
        this.serverIdSupplier = serverIdSupplier;
        this.aggregator = aggregator;
        this.startupLatch = startupLatch;
        this.stopAfter = stopAfter;
        YPath rootPath = YPath.simple(syncConfig.rootPath());
        this.syncConfig = syncConfig;
        this.importAllTables = this.syncConfig.importAllTables();
        this.alterHandler = this.syncConfig.importAllTables()
                ? new AlterHandler(yt, rootPath, rootPath, "straight", true) : null;
        this.excludeTableNames = syncConfig.getExcludeTableNames();
        this.verifyChecksums = syncConfig.verifyChecksums();
    }

    private void mySqlSettingsChanged() {
        settingsChanged = true;
    }

    /**
     * Создаёт временную директорию для работы mysqld
     */
    private static TempDirectory createMysqlTempDirectory(String prefix) {
        Path preferredPath = Paths.get("/dev/shm");
        if (preferredPath.toFile().isDirectory() && preferredPath.toFile().canWrite()) {
            // Если у нас есть права записи в /dev/shm, то создаём директорию там
            return new TempDirectory(preferredPath, prefix);
        } else {
            return new TempDirectory(prefix);
        }
    }

    public MysqlTransactionAggregator getAggregator() {
        return aggregator;
    }

    @Override
    public void run() {
        aggregator.lockAndRun(this::runInLock);
    }

    /**
     * Получение настроек подключения может выдавать SQLException, если используются слейвы
     * (за ними ходит в MySQL). Поэтому делаем некоторое количество ретраев.
     */
    private MySqlConnectionSettings getSettingsForSyncWithRetries(String dbName) {
        int timeoutMs = 60_000;
        int intervalMs = 100;
        Stopwatch sw = Stopwatch.createStarted();
        while (sw.elapsed(TimeUnit.MILLISECONDS) < timeoutMs) {
            try {
                return connectionProvider.getSettingsForSync(dbName);
            } catch (SQLException e) {
                logger.error(String.format("Exception when trying to get settings for %s, retrying...", dbName), e);
            }
            try {
                Thread.sleep(intervalMs);
            } catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                throw new InterruptedRuntimeException("Thread was interrupted", e);
            }
            intervalMs += intervalMs / 10;
        }
        throw new RuntimeException("Can't get connection settings: timeout waiting");
    }

    private void runInLock(AggregatorLock lock) {
        // Полное обновление схемы зеркализируемых таблиц исключает ситуацию,
        // когда таблица была создана одним инстансом зеркализатора,
        // который в дальнейшем теряет блокировку в YT,
        // а зеркализация продолжается на другом инстансе,
        // который еще не знает про новую таблицу
        // Иначе бинлоги новой таблицы будут проигнорированы
        // TODO : мб убрать проверку ?
        if (syncConfig.importAllTables()) {
            aggregator.refreshAllSchemas();
        }
        connectionProvider.addListener(listener);

        // При первом запуске false, при перезапуске true
        boolean restart = false;
        do {
            logger.info("Starting a temporary schema server...");
            // Если оно далее поменяется на true, это не страшно, т.к. мы просто выполним ещё одну итерацию
            settingsChanged = false;
            MySqlConnectionSettings settingsForSync = getSettingsForSyncWithRetries(dbName);
            //
            try (TempDirectory schemaServerDirectory = createMysqlTempDirectory("mysql-server-yt-sync")) {
                awaitAggregatorReady();

                MySQLServerBuilder schemaServerBuilder = new MySQLServerBuilder();
                schemaServerBuilder.setDataAndConfigDir(schemaServerDirectory.getPath());
                schemaServerBuilder.initializeDataDir();
                try (MySQLServer schemaServer = schemaServerBuilder.start()) {
                    MySQLBinlogState initialState = aggregator.getCurrentAggregatorState(true);
                    logger.info("Initialized position '{}'", initialState.getGtidSet());
                    BinlogEventConverter eventConverter = new BinlogEventConverter(initialState);
                    try (Connection schemaConnection = schemaServer.connect()) {
                        logger.info("Attaching schema server to the converter...");
                        eventConverter.attachSchemaConnection(schemaConnection);
                        if (startupLatch != null && !restart) {
                            // В первый раз сообщаем, что мы готовы к обработке данных и ждём остальные потоки
                            startupLatch.countDown();
                            startupLatch.await();
                        }
                        // true, если processEvents() обнаружил, что во время обработки произошло изменение dbConfig'а
                        restart = processEvents(settingsForSync, eventConverter, lock);
                    }
                }
            } catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                throw new InterruptedRuntimeException(e);
            } catch (IOException | SQLException | BinlogDDLException e) {
                throw new FatalException(e);
            }
            // На изменения dbConfig'а реагируем рестартом синхронизации
        } while (restart);
    }

    private void awaitAggregatorReady() throws InterruptedException {
        logger.info("Waiting for transaction aggregator to prepare");
        while (!aggregator.isReady()) {
            if (aggregator.isClosed()) {
                throw new IllegalStateException("Transaction aggregator is closed");
            }
            logger.info("Transaction aggregator is not ready yet, sleeping for 1s");
            TimeUnit.SECONDS.sleep(1);
        }
    }

    private boolean processEvents(MySqlConnectionSettings mySqlConnectionSettings,
                                  BinlogEventConverter eventConverter,
                                  AggregatorLock lock)
            throws IOException, InterruptedException, SQLException, BinlogDDLException {
        List<YtPendingOperation> currentTransaction = new ArrayList<>();
        // TODO : убрать это, когда разберёмся с тем, откуда возникают UPDATE после DELETE
        List<BinlogEvent> currentTransactionEvents = new ArrayList<>();
        boolean isResharding = false;
        while (true) {
            if (aggregator.isClosed() || !lock.isLocked()) {
                return false;
            }

            String host = mySqlConnectionSettings.getHosts().get(0);
            logger.info("Starting binlog event source from {}:{} at '{}'",
                    host, mySqlConnectionSettings.getPort(), eventConverter.getState().getGtidSet()
            );
            try (BinlogRawEventSource rawEventSource = new BinlogRawEventServerSource(
                    host,
                    mySqlConnectionSettings.getPort(),
                    mySqlConnectionSettings.getUser(),
                    mySqlConnectionSettings.getPass(),
                    serverIdSupplier.getServerId(),
                    eventConverter.getState().getGtidSet(),
                    32
            )) {
                eventConverter.attachRawEventSource(rawEventSource);
                logger.info("Starting to read binlog events...");
                while (true) {
                    if (aggregator.isClosed() || !lock.isLocked()) {
                        return false;
                    }
                    // Если запрошена пауза, то мы ожидаем, пока все данные будут записаны в yt,
                    // и после этого рапортуем о том, что синхронизация встала на паузу
                    if (pauseRequested) {
                        pauseLock.lock();
                        try {
                            pauseRequested = false;
                            logger.info("Syncing pause requested, waiting for pending data being flushed..");
                            aggregator.awaitPendingIsFlushed();

                            logger.info("Syncing is paused, current position is '{}'",
                                    aggregator.getCurrentAggregatorState().getGtidSet());

                            paused = true;
                            pauseCond.signalAll();
                            continue;
                        } finally {
                            pauseLock.unlock();
                        }
                    }
                    // Если пауза в действии, то просто ожидаем некоторое время и возвращаемся на начало цикла
                    if (paused) {
                        logger.info("Syncing is paused, continue waiting for resume");
                        Thread.sleep(5_000);
                        continue;
                    }
                    // Если во время работы были обновлены настройки подключения к этому шарду,
                    // выходим для попытки рестарта на новых настройках
                    if (settingsChanged) {
                        settingsChanged = false;
                        MySqlConnectionSettings newMySqlConnectionSettings = getSettingsForSyncWithRetries(dbName);
                        if (!newMySqlConnectionSettings.isEqualTo(mySqlConnectionSettings)) {
                            logger.warn("MySqlConnectionSettings change detected: was {}, now {}",
                                    mySqlConnectionSettings.toString(), newMySqlConnectionSettings.toString());
                            aggregator.awaitPendingIsFlushed();
                            logger.info("All transactions are flushed, current position is {}",
                                    aggregator.getCurrentAggregatorState().getGtidSet());
                            return true;
                        }
                    }

                    BinlogEvent event;
                    try {
                        event = eventConverter.readEvent(60, TimeUnit.SECONDS);
                    } catch (IOException | TimeoutException e) {
                        logger.error("Binlog connection failed, reconnecting in 15 seconds...", e);
                        TimeUnit.SECONDS.sleep(15);
                        break;
                    }
                    if (event == null) {
                        logger.error("Binlog stream disconnected, reconnecting immediately...");
                        break;
                    }
                    if (aggregator.isClosed()) {
                        throw new IllegalStateException("Transaction aggregator is closed");
                    }

                    switch (event.getType()) {
                        case DDL:
                            if (importAllTables) {
                                List<ru.yandex.direct.binlog.model.BinlogEvent> binlogEvents = ddlToBinlogEvent(dbName,
                                        dbName.split(":")[0], event)
                                        .stream()
//                                                .filter(e -> excludeTableNames.stream()
//                                                        .noneMatch(exclude -> e.getTable().matches(exclude)))
                                        .collect(Collectors.toList());
                                List<YPath> pathsToReload = new ArrayList<>();
                                for (ru.yandex.direct.binlog.model.BinlogEvent binlogEvent : binlogEvents) {
                                    try {
                                        aggregator.awaitPendingIsFlushed();
                                        List<YPath> tablesToMount = alterHandler.handle(binlogEvent,
                                                DummyStateManager.SHARD_OFFSET_SAVER);
                                        pathsToReload.addAll(tablesToMount);
                                    } catch (InterruptedException ex) {
                                        Thread.currentThread().interrupt();
                                        throw new InterruptedRuntimeException();
                                    } catch (TimeoutException ex) {
                                        throw new RuntimeTimeoutException("While handling " + binlogEvent);
                                    }
                                }
                                aggregator.addTransaction(event.getTimestamp(),
                                        new YtPendingTransaction(event.getData().getGtid(), currentTransaction),
                                        eventConverter.getState(), () -> logEvents(currentTransactionEvents));
                                currentTransaction.clear();
                                currentTransactionEvents.clear();
                                if (!pathsToReload.isEmpty()) {
                                    aggregator.refreshSchemaOnPaths(pathsToReload);
                                }
                            }
                        case DML:
                        case BEGIN:
                            // Нам эти события не интересны
                            break;
                        case ROWS_QUERY:
                            var rowsQuery = (BinlogEventData.RowsQuery) event.getData();
                            if (!isResharding) {
                                // Если в транзакции хотя бы один запрос с пометкой "решардинг", то считаем, что вся транзакция состоит из запросов решардинга.
                                // На момент внесения изменений так и есть, если станет не так, то надо отдельно решать, как такое обрабатывать
                                var traceInfo = DirectTraceInfo.extractIgnoringErrors(rowsQuery.getData().getQuery());
                                isResharding = traceInfo.getResharding();
                            }
                            break;
                        case ROLLBACK:
                            // После реконнекта нужно "откатить" текущую транзакцию
                            currentTransaction.clear();
                            currentTransactionEvents.clear();
                            isResharding = false;
                            break;
                        case INSERT: {
                            var insert = (BinlogEventData.Insert) event.getData();
                            String table = insert.getTableMap().getTable();
                            boolean skipInsert = false;
                            if (importAllTables && verifyChecksums && table.equals(YT_CHECKSUM)) {
                                // Инсерты в yt_checksum обрабатываются специальным образом
                                // А именно: на стороне yt выполняются аналогичные вычисления чексуммы
                                // и результат добавляется к вставляемой записи
                                // Это позволяет выполнять сверки данных
                                if (!currentTransaction.isEmpty()) {
                                    logger.error("Current transaction is not empty! Size is {}",
                                            currentTransaction.size());
                                }
                                MySQLSimpleRows rows = insert.getRows();
                                for (MySQLSimpleRow row : rows) {
                                    ChecksumRecord checksum = readChecksumRow(row);
                                    skipInsert = aggregator.verifyChecksum(checksum, event.getTimestamp());
                                }
                            }
                            if (!skipInsert) {
                                currentTransaction.add(new YtPendingInsert(event.getData(), dbName));
                                currentTransactionEvents.add(event);
                            }
                            break;
                        }
                        case UPDATE: {
                            currentTransaction.add(new YtPendingUpdate(event.getData(), dbName));
                            currentTransactionEvents.add(event);
                            break;
                        }
                        case DELETE:
                            currentTransaction.add(new YtPendingDelete(event.getData(), dbName));
                            currentTransactionEvents.add(event);
                            break;
                        case COMMIT: {
                            BinlogEventData.Commit data = event.getData();
                            if (stopAfter != null && !new GtidSet(data.getGtid()).isContainedWithin(stopAfter)) {
                                // Обнаружена транзакция, выходящая за рамки stopAfter
                                logger.info("Stopping without applying transaction {}", data.getGtid());
                                return false;
                            }
                            if (!isResharding || syncConfig.processReshardingEvents()) {
                                aggregator.addTransaction(event.getTimestamp(),
                                        new YtPendingTransaction(data.getGtid(), currentTransaction),
                                        eventConverter.getState(), () -> logEvents(currentTransactionEvents));
                            } else {
                                logger.info("Skipping transaction {} as resharding", data.getGtid());
                            }
                            currentTransaction.clear();
                            currentTransactionEvents.clear();
                            isResharding = false;
                            break;
                        }
                        default:
                            throw new IllegalStateException("Received unexpected event " + event);
                    }
                }
            }
        }
    }

    private ChecksumRecord readChecksumRow(MySQLSimpleRow after) {
        String tbl = null;
        Integer iteration = null;
        Integer chunk = null;
        Integer parentChunk = null;
        String boundaries = null;
        ChecksumMethod method = null;
        String crc = null;
        Integer crcCnt = null;
        String str = null;
        String nullsCnt = null;
        String columnsUsed = null;
        ChecksumStatus status = null;
        Timestamp ts = null;
        for (MySQLColumnData columnData : after) {
            switch (columnData.getSchema().getName()) {
                case "tbl":
                    tbl = (String) columnData.getRawValue();
                    break;
                case "iteration":
                    iteration = (Integer) columnData.getRawValue();
                    break;
                case "chunk":
                    chunk = (Integer) columnData.getRawValue();
                    break;
                case "parent_chunk":
                    parentChunk = (Integer) columnData.getRawValue();
                    break;
                case "boundaries":
                    boundaries = (String) columnData.getRawValue();
                    break;
                case "method":
                    method = ChecksumMethod.fromSource(columnData.getValueAsString());
                    break;
                case "this_crc":
                    crc = (String) columnData.getRawValue();
                    break;
                case "this_crc_cnt":
                    crcCnt = (Integer) columnData.getRawValue();
                    break;
                case "this_str":
                    str = columnData.getValueAsString();
                    break;
                case "this_nulls_cnt":
                    nullsCnt = (String) columnData.getRawValue();
                    break;
                case "columns_used":
                    columnsUsed = (String) columnData.getRawValue();
                    break;
                case "status":
                    status = ChecksumStatus.fromSource(columnData.getValueAsString());
                    break;
                case "ts":
                    ts = (Timestamp) columnData.getRawValue();
                    break;
            }
        }
        if (tbl == null || iteration == null || chunk == null || parentChunk == null || boundaries == null
                || method == null || columnsUsed == null || status == null || ts == null) {
            throw new RuntimeException("Incorrect checksum binlog received");
        }
        return new ChecksumRecord(tbl, iteration, chunk, parentChunk, boundaries, method,
                crc, crcCnt, str, nullsCnt, columnsUsed, status, ts);
    }

    private void logEvents(List<BinlogEvent> binlogEvents) {
        String binlogs = binlogEvents.stream().map(ev -> String.format("[\"%d\", \"%s\", \"%s\"]",
                ev.getTimestamp(), ev.getType(), ev.getData().getGtid())).collect(Collectors.joining(", "));
        logger.error("Consistency violation sequence: " + binlogs);
    }

    @Override
    public void close() {
        connectionProvider.removeListener(listener);
    }

    boolean isAliveWithAggregator() {
        return this.isAlive() && this.aggregator.isFlusherThreadAlive();
    }

    private static List<ru.yandex.direct.binlog.model.BinlogEvent> ddlToBinlogEvent(String source, String desiredDb,
                                                                                    ru.yandex.direct.mysql.BinlogEvent sourceBinlogEvent) {
        Preconditions.checkState(sourceBinlogEvent.getData() instanceof BinlogEventData.DDL);
        BinlogEventData.DDL ddl = sourceBinlogEvent.getData();
        try {
            String gtid = sourceBinlogEvent.getData().getGtid();

            ru.yandex.direct.binlog.model.BinlogEvent resultBinlogEvent = createDDLBinlogEventDraft(source, desiredDb,
                    sourceBinlogEvent, gtid);

            // rename table нельзя определить по схеме, он или не определится вовсе (например, если две таблицы
            // меняются местами). Или (что ещё хуже) определится как drop table + create table. Поэтому эта
            // проверка должна быть в начале.
            List<Pair<String, String>> renames = MysqlUtil.looksLikeRenameTable(ddl.getData().getSql());
            if (renames != null) {
                RenameTable renameTable = new RenameTable();
                boolean firstRename = true;
                for (Pair<String, String> rename : renames) {
                    if (firstRename) {
                        firstRename = false;
                        // на самом деле не используется, вставляется, чтобы механизм валидации корректно работал
                        resultBinlogEvent.setTable(rename.getLeft());
                    }
                    renameTable.withAddRename(rename.getLeft(), rename.getRight());
                }
                return Collections.singletonList(resultBinlogEvent.withAddedSchemaChanges(renameTable).validate());
            }

            Map<String, CreateTable> oldTables = MysqlUtil.tableByNameForDb(desiredDb, ddl.getBefore());
            Map<String, CreateTable> newTables = MysqlUtil.tableByNameForDb(desiredDb, ddl.getAfter());

            int changedTables = MysqlUtil.createTableEvent(resultBinlogEvent, oldTables, newTables)
                    + MysqlUtil.dropTableEvent(resultBinlogEvent, oldTables, newTables)
                    + MysqlUtil.columnChangeEvent(resultBinlogEvent, oldTables, newTables);
            if (changedTables == 0) {
                // truncate table невозможно определить через diff схемы
                // Эта проверка проводится после проверки diff'а, т.к. реализованный парсер ненадёжен,
                // чем меньше шансов ложного срабатывания, тем лучше.
                String truncatedTableName = MysqlUtil.looksLikeTruncateTable(ddl.getData().getSql());
                if (truncatedTableName != null) {
                    // truncate реализован как последовательность drop table и create table
                    ru.yandex.direct.binlog.model.BinlogEvent dropTableEvent =
                            createDDLBinlogEventDraft(source, desiredDb, sourceBinlogEvent, gtid)
                                    .withTable(truncatedTableName)
                                    .withSchemaChanges(Collections.singletonList(new DropTable(truncatedTableName)));
                    ru.yandex.direct.binlog.model.BinlogEvent createTableEvent =
                            createDDLBinlogEventDraft(source, desiredDb, sourceBinlogEvent, gtid)
                                    .withTable(truncatedTableName)
                                    .withAddedSchemaChanges(oldTables.get(truncatedTableName));
                    return Arrays.asList(dropTableEvent, createTableEvent);
                }
            }
            if (changedTables == 1 && !resultBinlogEvent.getSchemaChanges().isEmpty()) {
                List<Pair<String, String>> pairs = MysqlUtil.looksLikeRenameColumn(ddl.getData().getSql());
                if (pairs != null) {
                    List<SchemaChange> renameList = mapList(pairs, pair -> new RenameColumn()
                            .withOldColumnName(pair.getLeft())
                            .withNewColumnName(pair.getRight()));
                    // do not drop renamed columns in order to pass validation
                    List<String> shouldNotDrop = mapList(pairs, Pair::getLeft);
                    List<SchemaChange> schemaChanges = filterList(resultBinlogEvent.getSchemaChanges(),
                            schemaChange -> !(schemaChange instanceof DropColumn
                                    && shouldNotDrop.contains(((DropColumn) schemaChange).getColumnName())));
                    // renames should always go first in case of simultaneous column type modification
                    List<SchemaChange> readyToUse = Stream.concat(renameList.stream(), schemaChanges.stream())
                            .collect(Collectors.toList());
                    resultBinlogEvent.withSchemaChanges(readyToUse);
                }
                return Collections.singletonList(resultBinlogEvent.validate());
            } else {
                logger.warn("Got alter that changes {} tables. Skipped it. SQL: {}",
                        changedTables, ddl.getData().getSql());
                return Collections.emptyList();
            }
        } catch (RuntimeException exc) {
            throw new IllegalStateException("While handling SQL-query: " + ddl.getData().getSql(), exc);
        }
    }

    private static ru.yandex.direct.binlog.model.BinlogEvent createDDLBinlogEventDraft(String source,
                                                                                       String desiredDb,
                                                                                       BinlogEvent sourceBinlogEvent,
                                                                                       String gtid) {
        return new ru.yandex.direct.binlog.model.BinlogEvent()
                .withDb(desiredDb)
                .withGtid(gtid)
                .withOperation(Operation.SCHEMA)
                // Каждый DDL-запрос делается в отдельном gtid.
                .withQueryIndex(0)
                .withSource(source)
                .withUtcTimestamp(Instant.ofEpochMilli(sourceBinlogEvent.getTimestamp()).atZone(ZoneOffset.UTC)
                        .toLocalDateTime());
    }

    /**
     * Запрашивает паузу синхронизации.
     * После этого вызова данные ещё некоторое время дописываются в yt.
     * Чтобы подождать, пока это завершится, нужно позвать {@link #awaitSyncPaused}
     * Если же стример ещё не получил лок, то пауза будет получена только после взятия лока
     * (т.к. обработчик pauseRequested находится внутри processEvents)
     */
    public void requestPauseSync() {
        pauseLock.lock();
        try {
            if (!isAlive()) {
                throw new RuntimeException("Streamer thread has already finished");
            }
            if (paused) {
                return;
            }
            logger.info("Pausing sync");
            pauseRequested = true;
        } finally {
            pauseLock.unlock();
        }
    }

    /**
     * Ожидает момента фактической паузы синхронизации, когда все текущие данные записаны в yt,
     * а новые данные из бинлогов не читаются.
     */
    public void awaitSyncPaused() throws InterruptedException {
        pauseLock.lock();
        try {
            while (!paused) {
                if (!isAlive()) {
                    throw new RuntimeException("Streamer thread has already finished");
                }
                pauseCond.await(1, TimeUnit.SECONDS);
            }
        } finally {
            pauseLock.unlock();
        }
    }

    /**
     * Возобновляет синхронизацию.
     */
    public void resumeSync() {
        logger.info("Resuming sync");
        paused = false;
    }
}
