package ru.yandex.direct.mysql.ytsync.synchronizator.streamer.mysql;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.GregorianCalendar;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CountDownLatch;

import javax.annotation.ParametersAreNonnullByDefault;

import com.github.shyiko.mysql.binlog.GtidSet;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.mysql.MySQLBinlogState;
import ru.yandex.direct.mysql.ytsync.common.util.MySqlConnectionProvider;
import ru.yandex.direct.mysql.ytsync.synchronizator.snapshot.SnapshotService;
import ru.yandex.direct.mysql.ytsync.synchronizator.util.SyncConfig;
import ru.yandex.direct.mysql.ytsync.synchronizator.util.YtSyncUtil;
import ru.yandex.inside.yt.kosher.Yt;

@ParametersAreNonnullByDefault
public class MysqlTransactionStreamersPool implements AutoCloseable {
    private static final Logger logger = LoggerFactory.getLogger(MysqlTransactionStreamersPool.class);

    private final SyncConfig syncConfig;
    private final MysqlTransactionAggregatorFactory aggregatorFactory;
    private final MySqlConnectionProvider mySqlConnectionProvider;
    private final Yt yt;
    private final SnapshotService snapshotService;

    private final List<MysqlTransactionStreamer> threads = new ArrayList<>();
    private volatile boolean stopped;

    public MysqlTransactionStreamersPool(
            SyncConfig syncConfig,
            MysqlTransactionAggregatorFactory aggregatorFactory,
            MySqlConnectionProvider mySqlConnectionProvider,
            Yt yt,
            SnapshotService snapshotService) {
        if (syncConfig.getDbNames().isEmpty()) {
            throw new IllegalArgumentException("At least one dbName is required");
        }
        this.syncConfig = syncConfig;
        this.aggregatorFactory = aggregatorFactory;
        this.mySqlConnectionProvider = mySqlConnectionProvider;
        this.yt = yt;
        this.snapshotService = snapshotService;
    }

    private GtidSet getStopAfterSet(String dbName) {
        return Optional.ofNullable(syncConfig.streamStopAfterSet(dbName)).map(GtidSet::new).orElse(null);
    }

    public void syncInPool(AggregatorLock lock) throws InterruptedException {
        if (stopped) {
            throw new IllegalStateException("Cannot start a stopped streamer");
        }
        if (!threads.isEmpty()) {
            throw new IllegalStateException("This streamer has been started already");
        }
        CountDownLatch startupLatch = new CountDownLatch(syncConfig.getDbNames().size());
        for (String dbName : syncConfig.getDbNames()) {
            MysqlTransactionAggregator aggregator = aggregatorFactory.getAggregator(dbName);
            MySQLBinlogState initialState = aggregator.getCurrentAggregatorState();
            if (initialState == null) {
                throw new IllegalStateException(
                        "Current state for " + dbName + " is unknown, initial import is needed");
            }

            GtidSet stopAfter = getStopAfterSet(dbName);
            MysqlTransactionStreamer thread = new MysqlTransactionStreamer(
                    mySqlConnectionProvider,
                    dbName,
                    syncConfig::mysqlServerId,
                    aggregator,
                    startupLatch,
                    stopAfter,
                    syncConfig, yt);
            thread.setDaemon(true);
            threads.add(thread);
        }
        for (Thread thread : threads) {
            logger.info("Starting {}", thread.getName());
            thread.start();
        }

        waitForFinishAndControl();
    }

    /**
     * Чтобы не слишком часто проверять условия по выполнению снепшота, делаем это не чаще раза в минуту
     */
    private boolean canCheckSnapshot() {
        GregorianCalendar calendar = new GregorianCalendar();
        return calendar.get(Calendar.SECOND) == 0;
    }

    private void waitForFinishAndControl() throws InterruptedException {
        while (true) {
            Thread.sleep(1000);

            // Если требуется сделать снепшот -- паузим все стримеры, выполняем степшот, возобновляем синхронизацию
            try {
                if (canCheckSnapshot() && snapshotService.needSnapshotNow()) {
                    try {
                        logger.info("Pausing all streamers");
                        for (MysqlTransactionStreamer thread : threads) {
                            thread.requestPauseSync();
                        }
                        logger.info("Waiting for all streamers to be paused");
                        for (MysqlTransactionStreamer thread : threads) {
                            thread.awaitSyncPaused();
                        }
                        logger.info("Creating snapshot");
                        //
                        snapshotService.createSnapshot();
                        //
                        logger.info("Creating snapshot finished");
                    } finally {
                        logger.info("Resuming all streamers");
                        for (MysqlTransactionStreamer thread : threads) {
                            thread.resumeSync();
                        }
                    }
                }
            } catch (RuntimeException e) {
                logger.error("Error when executing snapshot logic", e);
            }

            for (MysqlTransactionStreamer thread : threads) {
                if (thread.isAliveWithAggregator()) {
                    continue;
                }
                // Если один из потоков умер, мы должны завершить работу
                logger.info("One of threads is dead, terminating all other threads");
                close();
                return;
            }
            if (stopped) {
                return;
            }
        }
    }

    @Override
    public void close() {
        if (stopped) {
            logger.warn("Already stopped.");
            return;
        }
        List<MysqlTransactionStreamer> backgroundThreads;
        synchronized (this) {
            backgroundThreads = new ArrayList<>(threads);
            stopped = true;
            threads.clear();
        }
        for (MysqlTransactionStreamer thread : backgroundThreads) {
            logger.info("Stopping {}", thread.getName());
            thread.getAggregator().close();
            thread.interrupt();
        }
        for (MysqlTransactionStreamer thread : backgroundThreads) {
            logger.info("Waiting for {} to finish", thread.getName());
            YtSyncUtil.doUninterruptibly(thread::join); // IS-NOT-COMPLETABLE-FUTURE-JOIN
            thread.close();
        }
    }
}
