package ru.yandex.direct.mysql.ytsync.synchronizator.streamer.yt;

import java.time.Duration;
import java.util.Timer;
import java.util.TimerTask;
import java.util.function.Consumer;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.mysql.ytsync.synchronizator.streamer.mysql.AggregatorLock;
import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.transactions.Transaction;

/**
 * Класс для наблюдения за транзакцией, под которой был взят лок. Если транзакция отвалилась - класс меняет статус лока
 * TODO : rename to YtLockWatcher, there is not Aggregator specifics
 */
public class YtAggregatorLockWatcher implements AggregatorLock, AutoCloseable {
    private static final Logger logger = LoggerFactory.getLogger(YtAggregatorLockWatcher.class);
    private static final Duration EXECUTION_PERIOD = Duration.ofSeconds(20);
    private final Timer timer;
    private final Thread threadToInterrupt;
    private volatile boolean transactionIsActive;
    private volatile boolean threadInterrupted = false;

    public YtAggregatorLockWatcher(Yt yt, Transaction transaction) {
        this(yt, transaction, null);
    }

    public YtAggregatorLockWatcher(Yt yt, Transaction transaction, Thread threadToInterrupt) {
        transactionIsActive = true;
        this.threadToInterrupt = threadToInterrupt;
        this.timer = new Timer(true);

        TransactionChecker task = new TransactionChecker(yt, transaction, this::setTransactionIsActive);
        long period = EXECUTION_PERIOD.toMillis();
        this.timer.scheduleAtFixedRate(task, period, period);
    }

    private void setTransactionIsActive(boolean transactionIsActive) {
        this.transactionIsActive = transactionIsActive;
        if (!transactionIsActive && threadToInterrupt != null) {
            if (!threadInterrupted) {
                logger.error("The lock is lost, interrupting thread");
                threadToInterrupt.interrupt();
                threadInterrupted = true;
            } else {
                logger.warn("The lock is lost, thread interrupted already");
            }
        }
    }

    @Override
    public boolean isLocked() {
        return transactionIsActive;
    }

    @Override
    public void close() {
        timer.cancel();
    }

    public static class TransactionChecker extends TimerTask {
        private final YPath transactionPath;
        private final Consumer<Boolean> transactionIsActiveConsumer;
        private final Yt yt;

        public TransactionChecker(Yt yt, Transaction transaction,
                                  Consumer<Boolean> transactionIsActiveConsumer) {
            this.yt = yt;
            this.transactionIsActiveConsumer = transactionIsActiveConsumer;
            this.transactionPath = YPath.simple(String.format("//sys/transactions/%s", transaction.getId()));
        }

        @Override
        public void run() {
            try {
                // TODO : Сделать несколько ретраев, и если не получилось, то пометить транзакцию как потерянную
                transactionIsActiveConsumer.accept(isTransactionActive());
            } catch (RuntimeException e) {
                logger.error("Unhandled exception in TransactionChecker timer task, ignore it", e);
            } catch (Throwable e) {
                logger.error("Unhandled throwable in TransactionChecker timer task, thread will be finished", e);
                throw e;
            }
        }

        private boolean isTransactionActive() {
            boolean isActive = yt.cypress().exists(transactionPath);
            logger.debug("Transaction {} is active: {}", transactionPath, isActive);
            return isActive;
        }
    }
}
