package ru.yandex.direct.mysql.ytsync.synchronizator.streamer.yt;

import java.math.BigDecimal;
import java.math.MathContext;
import java.math.RoundingMode;
import java.nio.charset.StandardCharsets;
import java.text.DecimalFormat;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.BiFunction;
import java.util.function.BooleanSupplier;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import java.util.zip.CRC32;

import javax.annotation.Nullable;

import com.google.common.base.Preconditions;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.binlogbroker.replicatetoyt.YtReplicator;
import ru.yandex.direct.env.EnvironmentType;
import ru.yandex.direct.mysql.MySQLBinlogState;
import ru.yandex.direct.mysql.schema.ColumnSchema;
import ru.yandex.direct.mysql.schema.DatabaseSchema;
import ru.yandex.direct.mysql.schema.KeySchema;
import ru.yandex.direct.mysql.schema.TableSchema;
import ru.yandex.direct.mysql.ytsync.common.compatibility.YtLockUtil;
import ru.yandex.direct.mysql.ytsync.common.compatibility.YtSupport;
import ru.yandex.direct.mysql.ytsync.common.components.SyncStatesTable;
import ru.yandex.direct.mysql.ytsync.common.row.FlatRow;
import ru.yandex.direct.mysql.ytsync.common.util.YtSyncCommonUtil;
import ru.yandex.direct.mysql.ytsync.synchronizator.operation.YtPendingTransaction;
import ru.yandex.direct.mysql.ytsync.synchronizator.streamer.mysql.AggregatorLock;
import ru.yandex.direct.mysql.ytsync.synchronizator.streamer.mysql.ChecksumMethod;
import ru.yandex.direct.mysql.ytsync.synchronizator.streamer.mysql.ChecksumRecord;
import ru.yandex.direct.mysql.ytsync.synchronizator.streamer.mysql.ChecksumStatus;
import ru.yandex.direct.mysql.ytsync.synchronizator.streamer.mysql.MysqlTransactionAggregator;
import ru.yandex.direct.mysql.ytsync.synchronizator.tableprocessors.TableProcessor;
import ru.yandex.direct.mysql.ytsync.synchronizator.tables.TableWriteSink;
import ru.yandex.direct.mysql.ytsync.synchronizator.util.SyncConfig;
import ru.yandex.direct.mysql.ytsync.synchronizator.util.YtSyncUtil;
import ru.yandex.direct.utils.InterruptedRuntimeException;
import ru.yandex.direct.utils.ThreadUtils;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTree;
import ru.yandex.yt.ytclient.tables.ColumnValueType;
import ru.yandex.yt.ytclient.wire.UnversionedRow;
import ru.yandex.yt.ytclient.wire.UnversionedRowset;
import ru.yandex.yt.ytclient.wire.UnversionedValue;

import static ru.yandex.direct.mysql.MySQLColumnType.DATE_TIME_FORMATTER;
import static ru.yandex.direct.mysql.ytsync.common.util.YtSyncCommonUtil.parseDbName;
import static ru.yandex.direct.mysql.ytsync.synchronizator.streamer.mysql.ChecksumRecord.YT_CHECKSUM;
import static ru.yandex.direct.mysql.ytsync.synchronizator.util.YtSyncUtil.getSourceColumnValue;

/**
 * Аггрегатор входных данных mysql на yt
 * <p>
 * Схема работы предполагает чтение данных из нескольких шардов и добавление их в этот класс. Добавление данных
 * потоко-безопасное и разделено по базам данных и шардам. Есть ограничение на размер буфферизируемых данных.
 */
public class YtTransactionAggregator implements MysqlTransactionAggregator {
    private static final Logger logger = LoggerFactory.getLogger(YtTransactionAggregator.class);

    // Если в потоке backgroundFlusher возникло необработанное исключение
    // (обычно это происходит из-за операции, которая так и не смогла успешно заретраиться в течение своего deadline),
    // то поток подождёт столько времени перед следующей попыткой зафлашить изменения в YT
    private static final Duration FLUSHER_RECOVER_INTERVAL = Duration.ofSeconds(10);

    private final EnvironmentType environmentType;
    private final String dbName;
    private final YPath lockPath;
    private final YtSupport ytSupport;
    private final SyncConfig syncConfig;
    private final SyncStatesTable syncStatesTable;
    private final Yt yt;
    private final YtCluster cluster;
    private final Map<String, MySQLBinlogState> pendingStates = new ConcurrentHashMap<>();
    private final Map<String, Long> pendingTimestamps = new ConcurrentHashMap<>();
    private final Function<String, List<TableProcessor>> tableProcessorsProvider;
    // Два TableSet'а меняются местами: пока первый применяется на yt во второй агрегируются данные
    private TableSet applyingSet;
    private TableSet pendingSet;
    private Lock pendingLock = new ReentrantLock(true);
    private Condition pendingIsFull = pendingLock.newCondition();
    private Condition pendingIsEmpty = pendingLock.newCondition();
    private Condition pendingIsFlushed = pendingLock.newCondition();
    private int pendingOperations = 0;
    private long firstBatchTimestamp = Long.MAX_VALUE;

    private volatile Thread flusher = null;
    private volatile boolean stopped = false;
    private volatile boolean forceFlush = false;
    private volatile boolean applyingSetFlushed = true;

    // Всё это настраивается в конфиге
    private boolean useRpcTransactions;
    private int maxPendingOperations;
    private int flushEveryMillis;
    private Duration globalProgressTimeout;
    private BiFunction<YPath, String, TableProcessor> yPathToTaskProcessorConverter;

    public YtTransactionAggregator(
            EnvironmentType environmentType,
            String dbName,
            YPath lockPath, YtSupport ytSupport,
            Yt yt,
            SyncConfig syncConfig,
            SyncStatesTable syncStatesTable,
            Function<String, List<TableProcessor>> tableProcessorsProvider,
            BiFunction<YPath, String, TableProcessor> yPathToTaskProcessorConverter) {
        this.environmentType = environmentType;
        this.dbName = dbName;
        this.lockPath = lockPath;
        this.ytSupport = ytSupport;
        this.syncConfig = syncConfig;
        this.syncStatesTable = syncStatesTable;
        this.yt = yt;
        this.cluster = syncConfig.cluster();
        this.tableProcessorsProvider = tableProcessorsProvider;
        refreshAllSchemas();
        this.yPathToTaskProcessorConverter = yPathToTaskProcessorConverter;
    }

    @Override
    public void lockAndRun(Consumer<AggregatorLock> consumer) {
        BooleanSupplier interruptedConsumer = () -> Thread.currentThread().isInterrupted() || stopped;
        YtLockUtil.runInLock(yt, lockPath, interruptedConsumer, transaction -> {
            if (interruptedConsumer.getAsBoolean()) {
                return;
            }

            try (YtAggregatorLockWatcher lockWatcher = new YtAggregatorLockWatcher(yt, transaction)) {
                consumer.accept(lockWatcher);
            }
        });
    }

    @Override
    public MySQLBinlogState getCurrentAggregatorState() {
        return getCurrentAggregatorState(false);
    }

    @Override
    public MySQLBinlogState getCurrentAggregatorState(boolean clearCacheBeforeLookup) {
        synchronized (syncStatesTable) {
            if (clearCacheBeforeLookup) {
                syncStatesTable.clear();
            }
            return syncStatesTable.getBinlogState(dbName);
        }
    }

    @Override
    public boolean isReady() {
        return true;
    }

    @Override
    public boolean isClosed() {
        return stopped;
    }

    @Override
    public void awaitPendingIsFlushed() throws InterruptedException {
        pendingLock.lockInterruptibly();
        try {
            // Устанавливаем флаг, запрещающий ожидание дополнительных событий
            // в pendingSet и добавление новых транзакций
            forceFlush = true;

            // Ожидаем, что pending станет пустым, и все данные будут свапнуты в applying
            // Цикл нужен для предотвращения гонок и одновременно для обработки spurious wakeup
            while (!pendingStates.isEmpty()) {
                if (isClosed()) {
                    throw new RuntimeException("Can't await pending to be flushed: flusher thread is stopped");
                }
                // Ожидаем с таймаутом, т.к. может оказаться, что сигнал не будет подан вовсе
                pendingIsEmpty.await(1, TimeUnit.SECONDS);
            }

            // Ожидаем, что весь applying будет записан в yt
            // Цикл нужен для предотвращения гонок и одновременно для обработки spurious wakeup
            while (!applyingSetFlushed) {
                if (isClosed()) {
                    throw new RuntimeException("Can't await pending to be flushed: flusher thread is stopped");
                }
                // Ожидаем с таймаутом, т.к. может оказаться, что сигнал не будет подан вовсе
                pendingIsFlushed.await(1, TimeUnit.SECONDS);
            }
        } finally {
            forceFlush = false;
            pendingLock.unlock();
        }
    }

    @Override
    public void refreshSchemaOnPaths(List<YPath> pathsToReload) {
        pendingLock.lock();
        try {
            Set<String> paths = pathsToReload.stream()
                    .map(YPath::toString)
                    .collect(Collectors.toSet());
            for (List<TableProcessor> tableProcessors : Arrays.asList(pendingSet.processors, applyingSet.processors)) {
                // remove paths
                List<TableProcessor> toRemove = tableProcessors.stream()
                        .filter(tableProcessor -> paths.contains(tableProcessor.getMainTable().getPath()))
                        .collect(Collectors.toList());
                tableProcessors.removeAll(toRemove);
                // add paths
                List<TableProcessor> toAdd = pathsToReload.stream()
                        .map(yPath -> yPathToTaskProcessorConverter.apply(yPath, dbName))
                        .collect(Collectors.toList());
                tableProcessors.addAll(toAdd);
            }
        } finally {
            pendingLock.unlock();
        }
    }

    /**
     * Обновляет состав и схему синхронизируемых таблиц
     */
    @Override
    public void refreshAllSchemas() {
        logger.info("Regenerating all table processors");
        this.applyingSet = new TableSet(tableProcessorsProvider.apply(dbName));
        this.pendingSet = new TableSet(tableProcessorsProvider.apply(dbName));
    }

    @Override
    public void addTransaction(long timestamp, YtPendingTransaction tx,
                               MySQLBinlogState state, Runnable consistencyViolationHandler)
            throws InterruptedException {
        pendingLock.lockInterruptibly();
        try {
            if (forceFlush) {
                throw new IllegalStateException("Cannot add transaction while waiting the data being flushed");
            }
            while (pendingOperations >= maxPendingOperations) {
                if (isClosed()) {
                    throw new RuntimeException(
                            "Can't wait some space in pending operations: flusher thread is stopped");
                }
                // Дожидаемся, пока в очереди будет меньше чем maxPendingOperations операций
                logger.info("Waiting for some space in pending operations...");
                // Тоже с таймаутом, на случай если backgroundFlusher будет остановлен, пока мы ждём здесь
                pendingIsEmpty.await(1, TimeUnit.MINUTES);
            }

            // Отдаём транзакцию во все процессоры и запоминаем обновлённое состояние
            boolean consistencyViolationFound = false;
            for (TableProcessor processor : pendingSet.processors) {
                pendingOperations += tx.apply(processor);
                if (processor.getSinks().stream().anyMatch(TableWriteSink::isConsistencyViolationFound)) {
                    consistencyViolationFound = true;
                }
            }
            pendingStates.put(dbName, state);
            pendingTimestamps.put(dbName, timestamp);
            firstBatchTimestamp = Math.min(firstBatchTimestamp, timestamp);

            if (consistencyViolationFound && consistencyViolationHandler != null) {
                consistencyViolationHandler.run();
            }

            if (pendingOperations >= maxPendingOperations) {
                // Сообщаем flush потоку, что пора бы уже делать flush
                pendingIsFull.signalAll();
            }
        } finally {
            pendingLock.unlock();
        }
    }

    private TakenOperationsInfo takePendingOperationsLocked() {
        TakenOperationsInfo result = new TakenOperationsInfo();
        result.opsCount = pendingOperations;
        result.firstBatchTimestamp = firstBatchTimestamp;
        result.takeStartNanoTime = System.nanoTime();

        // Меняем местами current и pending
        TableSet tmp = applyingSet;
        applyingSet = pendingSet;
        pendingSet = tmp;

        // Обнуляем статистику
        pendingOperations = 0;
        synchronized (syncStatesTable) {
            for (Map.Entry<String, MySQLBinlogState> entry : pendingStates.entrySet()) {
                syncStatesTable.setBinlogState(entry.getKey(), entry.getValue());
                syncStatesTable.setLastTimestamp(entry.getKey(), pendingTimestamps.get(entry.getKey()));
            }
        }
        pendingStates.clear();
        pendingTimestamps.clear();
        firstBatchTimestamp = Long.MAX_VALUE;

        pendingIsEmpty.signalAll();
        return result;
    }

    /**
     * Дожидается наполнения буфера операций и закватывает его для применения на yt
     *
     * @param timeoutNanos максимальная задержка от появления первой операции до захвата неполного буфера
     * @return информация о закваченных операциях
     */
    private TakenOperationsInfo waitForPendingOperations(long timeoutNanos) throws InterruptedException {
        TakenOperationsInfo result;
        long waitStartTime = System.nanoTime();
        long deadline = waitStartTime + timeoutNanos;

        pendingLock.lock();
        try {
            // Дожидаемся полного наполнения буфера транзакицями
            while (!forceFlush && !stopped && (pendingStates.isEmpty() || pendingOperations < maxPendingOperations)) {
                long delay = deadline - System.nanoTime();
                if (!pendingIsFull.await(delay, TimeUnit.NANOSECONDS)) {
                    // Мы не дождались наполнения буфера за время таймаута
                    if (pendingStates.isEmpty() && pendingOperations <= 0) {
                        // Увеличиваем таймаут, если по факту нам нечего применять
                        waitStartTime = System.nanoTime();
                        deadline = waitStartTime + timeoutNanos;
                        continue;
                    }
                    break;
                }
            }

            if (stopped) {
                return null;
            }

            result = takePendingOperationsLocked();
            applyingSetFlushed = false;
        } finally {
            pendingLock.unlock();
        }

        logger.info("Taken {} pending operations after {}ms...", result.opsCount,
                TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - waitStartTime));
        return result;
    }

    private CompletableFuture<Void> applyCurrentOperations(YtSupport.Transaction tx) {
        List<CompletableFuture<Void>> applyFutures = new ArrayList<>();
        Set<TableWriteSink> sinks = new HashSet<>();
        for (TableProcessor processor : applyingSet.processors) {
            processor.flush();
            sinks.addAll(processor.getSinks());
        }
        for (TableWriteSink sink : sinks) {
            applyFutures.add(sink.apply(tx));
        }
        return YtSyncCommonUtil.allFutures(applyFutures);
    }

    private void flushTaken(TakenOperationsInfo info) throws InterruptedException {
        long t2 = System.nanoTime();

        CompletableFuture<Void> applyFuture;
        if (useRpcTransactions) {
            applyFuture = ytSupport.runTransaction(this::applyCurrentOperations);
        } else {
            applyFuture = ytSupport.nullTransaction().thenCompose(this::applyCurrentOperations);
        }
        YtSyncUtil.joinInterruptibly(applyFuture);

        for (TableProcessor processor : applyingSet.processors) {
            for (TableWriteSink sink : processor.getSinks()) {
                sink.clear();
            }
        }

        long t3 = System.nanoTime();

        // К этому моменту данные уже есть в таблицах, независимо от длительности записи в sync-states
        long maxDelay =
                info.firstBatchTimestamp < Long.MAX_VALUE ? System.currentTimeMillis() - info.firstBatchTimestamp : 0;

        // Сохраняем состояние в отдельной транзакции
        synchronized (syncStatesTable) {
            YtSyncUtil.joinInterruptibly(ytSupport.runTransaction(syncStatesTable::apply));
            syncStatesTable.committed();
        }

        long tend = System.nanoTime();

        logger.info(
                "Flushed: max delay is {} ({} take, {} apply, {} total)",
                YtSyncUtil.formatMillis(maxDelay),
                YtSyncUtil.formatMillis(TimeUnit.NANOSECONDS.toMillis(t2 - info.takeStartNanoTime)),
                YtSyncUtil.formatMillis(TimeUnit.NANOSECONDS.toMillis(t3 - t2)),
                YtSyncUtil.formatMillis(TimeUnit.NANOSECONDS.toMillis(tend - info.takeStartNanoTime)));

        pendingLock.lock();
        try {
            applyingSetFlushed = true;
            pendingIsFlushed.signalAll();
        } finally {
            pendingLock.unlock();
        }
    }

    private void backgroundFlusher() throws InterruptedException {
        long lastFlushTime = System.nanoTime();
        boolean takenIsFlushed = true;
        TakenOperationsInfo info = null;
        while (!stopped) {
            try {
                long nextFlushTime = lastFlushTime + TimeUnit.MILLISECONDS.toNanos(flushEveryMillis);
                long currentTime = System.nanoTime();
                long timeoutNanos = nextFlushTime - currentTime;
                if (timeoutNanos < 0) {
                    // Похоже мы не успеваем делать flush
                    timeoutNanos = 0;
                }
                // пока текущий буфер не будет успешно накачен
                // не пытаемся забрать следующий
                if (takenIsFlushed) {
                    try {
                        info = waitForPendingOperations(timeoutNanos);
                    } catch (RuntimeException e) {
                        logger.error("Exception in waitForPendingOperations, exiting backgroundFlusher thread", e);
                        break;
                    }
                    if (info == null) {
                        logger.info("waitForPendingOperations returned null, exiting backgroundFlusher thread");
                        break;
                    }
                    takenIsFlushed = false;
                }
                flushTaken(info);
                lastFlushTime = info.takeStartNanoTime;
                takenIsFlushed = true;
            } catch (RuntimeException e) {
                logger.error("Exception in backgroundFlusher thread", e);

                long secondsAfterLastFlush =
                        TimeUnit.NANOSECONDS.toSeconds(Math.max(0, System.nanoTime() - lastFlushTime));
                // Если в течение значительного времени мы не получаем ничего, кроме ошибок,
                // то имеет смысл завершить поток (а следом и весь процесс, и дать шанс другому инстансу)
                if (secondsAfterLastFlush > globalProgressTimeout.getSeconds()) {
                    logger.error("No progress in {} seconds detected, thread will be terminated",
                            secondsAfterLastFlush);
                    stopped = true;
                } else {
                    logger.info("Sleep {} seconds before continue..", FLUSHER_RECOVER_INTERVAL.getSeconds());
                    Thread.sleep(FLUSHER_RECOVER_INTERVAL.toMillis());
                }
            }
        }
    }

    public void start() {
        if (flusher != null) {
            throw new IllegalStateException("Background flusher already started");
        }

        useRpcTransactions = syncConfig.useRpcTransactions();
        maxPendingOperations = syncConfig.maxPendingOperations();
        flushEveryMillis = syncConfig.flushEveryMillis();
        globalProgressTimeout = Duration.ofSeconds(syncConfig.globalProgressTimeoutSeconds());

        logger.info("Starting background flusher thread");
        flusher = new Thread(() -> {
            try {
                backgroundFlusher();
            } catch (InterruptedException e) {
                logger.warn("Background flusher interrupted", e);
            } catch (Throwable e) {
                // Если вдруг просочился Error, то хотя бы залогируем его
                logger.error("Unexpected exception in backgroundFlusher thread, exiting", e);
            }
        });
        flusher.setName("yt-binlog-aggregator " + dbName);
        flusher.setDaemon(true);
        flusher.start();
    }

    @Override
    public void close() {
        if (flusher != null) {
            stopped = true;
            // TODO: мб убрать interrupt(), уже ведь устанавливаем флаг stopped
            flusher.interrupt();
            boolean interrupted = false;
            try {
                logger.info("Waiting for flusher thread to stop");
                while (flusher.isAlive()) {
                    try {
                        flusher.join(); // IS-NOT-COMPLETABLE-FUTURE-JOIN
                        break;
                    } catch (InterruptedException ie) {
                        interrupted = true;
                    }
                }
            } finally {
                if (interrupted) {
                    Thread.currentThread().interrupt();
                }
            }
        }
    }

    @Override
    public boolean isFlusherThreadAlive() {
        Thread flusherThread = this.flusher;
        return flusherThread != null && flusherThread.isAlive();
    }

    /**
     * Выполняет преобразование boundaries в синтаксисе mysql к синтаксису yt
     * Для этого lowercased-колонки заменяются на точные их имена, а апострофы и кавычки удаляются
     * TODO : unit test, move to utils
     */
    private String unescape(String boundaries, TableSchema tableSchema) {
        String result = boundaries;
        for (ColumnSchema column : tableSchema.getColumns()) {
            result = result.replaceAll(column.getName().toLowerCase(), column.getName());
        }
        return result.replaceAll("`", "").replaceAll("'", "");
    }

    /**
     * Получает колонки первичного ключа и возвращает их в виде строки для включения в запрос
     * Пример: "ClientID, date"
     * TODO : unit test, move to utils
     */
    private String getOrderByPkSql(TableSchema tableSchema) {
        KeySchema primaryKeySchema = tableSchema.getPrimaryKey().orElseThrow(() ->
                new IllegalStateException("Table without Primary Key is not supported"));
        if (!primaryKeySchema.getType().equals("BTREE")) {
            throw new IllegalStateException(String.format("Unknown primary key type %s", primaryKeySchema.getType()));
        }
        Preconditions.checkState(!primaryKeySchema.getColumns().isEmpty());
        if (primaryKeySchema.getColumns().stream().anyMatch(col -> col.getPart() != null)) {
            throw new IllegalStateException("Indexes with sub_part is not supported");
        }
        return primaryKeySchema.getColumns().stream()
                .map(keyColumn -> {
                    ColumnSchema column = tableSchema.findColumn(keyColumn.getName());
                    if ("enum".equals(column.getDataType())) {
                        return convertEnumToOrderBy(column);
                    }
                    return keyColumn.getName();
                }).collect(Collectors.joining(", "));
    }

    private String convertEnumToOrderBy(ColumnSchema columnSchema) {
        // Преобразует значения в enum('client','brand','agency')
        // в строчку, которую можно использовать в качестве ORDER BY:
        // if(type = 'client', 1, if(type = 'brand', 2, 3))
        Preconditions.checkState(columnSchema.getDataType().equals("enum"));
        Pattern pattern = Pattern.compile("'(?<value>\\w+)'");
        Matcher matcher = pattern.matcher(columnSchema.getColumnType());
        List<String> enumValues = new ArrayList<>();
        // Пустая строка при сортировке является самой первой
        // (на самом деле есть ещё NULL, но ключевая колонка не может быть NULLABLE в mysql)
        enumValues.add("");
        while (matcher.find()) {
            enumValues.add(matcher.group("value"));
        }
        String orderBy = Integer.toString(enumValues.size());
        for (int i = enumValues.size() - 2; i >= 0; i--) {
            orderBy = String.format("if(%s = '%s', %d, %s)", columnSchema.getName(), enumValues.get(i), i + 1, orderBy);
        }
        return orderBy;
    }

    private List<ColumnSchema> getUsedColumns(TableSchema tableSchema, String usedMask) {
        List<ColumnSchema> columns = tableSchema.getColumns();
        List<ColumnSchema> used = new ArrayList<>();
        for (int i = 0; i < columns.size(); i++) {
            if (usedMask.charAt(i) == '1') {
                used.add(columns.get(i));
            }
        }
        return used;
    }

    // Проверяет, есть ли таблица в выгрузке или она была пропущена
    private boolean isTableImported(YPath tablePath) {
        for (TableProcessor processor : pendingSet.processors) {
            if (tablePath.toString().equals(processor.getMainTable().getPath())) {
                return true;
            }
        }
        return false;
    }

    private Pair<String, YtCluster> parseTableName(String tbl) {
        String[] split = tbl.split("/");
        return Pair.of(split[0], YtCluster.parse(split[1]));
    }

    // Если отставание превышает этот порог, проверка контрольных сумм не выполняется
    // (записи идут в yt_checksum как есть)
    private static final Duration VERIFY_CRC_LAG_THRESHOLD = Duration.ofMinutes(20);

    @Override
    public boolean verifyChecksum(ChecksumRecord checksum, long timestamp) {
        try {
            if (Duration.ofMillis(System.currentTimeMillis() - timestamp).compareTo(VERIFY_CRC_LAG_THRESHOLD) >= 0) {
                logger.info("Skip verifyChecksum (time lag is too large)");
                return false;
            }

            Pair<String, YtCluster> destination = parseTableName(checksum.getTable());
            String table = destination.getLeft();
            YtCluster ytCluster = destination.getRight();
            if (cluster != ytCluster) {
                return false;
            }

            YPath tablePath = YPath.simple(syncConfig.rootPath()).child(dbName).child("straight").child(table);

            if (!isTableImported(tablePath)) {
                logger.info("Skip verifyChecksum {} (table is not imported)", checksum);
                return false;
            }

            try {
                awaitPendingIsFlushed();
            } catch (InterruptedException e) {
                throw new InterruptedRuntimeException(e);
            }

            List<DatabaseSchema> databases;
            synchronized (syncStatesTable) {
                databases = syncStatesTable.lookupRow(dbName).orElseThrow().getServerSchema().getDatabases();
            }
            DatabaseSchema dbSchema = databases.stream()
                    .filter(db -> db.getName().equals(parseDbName(dbName).getDb()))
                    .findFirst().orElseThrow();
            TableSchema tableSchema = dbSchema.getTables().stream()
                    .filter(t -> t.getName().equals(table)).findFirst().orElseThrow();

            // Здесь можно было бы использовать настоящее кол-во таблетов,
            // но его нужно откуда-то получить. В нашем состоянии оно не хранится.
            // Поэтому проще пока указать все 64 таблета
            String allTabletsPredicate = String.format("%s in (%s)",
                    YtReplicator.HASH_COLUMN_NAME,
                    IntStream.range(0, 64)
                            .mapToObj(Integer::toString)
                            .collect(Collectors.joining(",")));

            String boundaries = checksum.getBoundaries();
            List<ColumnSchema> columns = getUsedColumns(tableSchema, checksum.getColumnsUsed());
            Preconditions.checkState(!columns.isEmpty());
            String query = String.format(
                    "%s FROM [%s] WHERE (%s) AND (%s) ORDER BY %s LIMIT 1000000",
                    columns.stream().map(ColumnSchema::getName).collect(Collectors.joining(", ")),
                    tablePath.toString(),
                    allTabletsPredicate,
                    unescape(boundaries, tableSchema),
                    getOrderByPkSql(tableSchema)
            );
            logger.info("Query: " + query);

            List<UnversionedRow> rows = executeQuery(query).getRows();
            if (checksum.getMethod() == ChecksumMethod.CRC) {
                CrcResult result = computeCrc(rows, columns);

                if (!Objects.equals(checksum.getCrcCnt(), result.cnt)
                        || !Objects.equals(checksum.getCrc(), result.crc)) {
                    logger.error("Crc mismatch! Chunk {}, got {}/{}, should be {}/{}", checksum.getChunk(),
                            result.cnt, result.crc, checksum.getCrcCnt(), checksum.getCrc());
                } else {
                    logger.info("Crc OK. Chunk {}, got {}/{}", checksum.getChunk(), result.cnt, result.crc);
                }

                // Записать результат в YT-таблицу
                insertYtChecksumResultRetrying(checksum, result.cnt, result.crc, null, ChecksumStatus.STARTED);
            } else if (checksum.getMethod() == ChecksumMethod.TO_STRING) {
                String str = rowsToString(rows, columns, nullableColumns(columns));
                if (!Objects.equals(checksum.getStr(), str)) {
                    logger.error("ToStr mismatch! Chunk {}, got {}, should be {}",
                            checksum.getChunk(), str, checksum.getStr());
                } else {
                    logger.error("ToStr OK! Chunk {}", checksum.getChunk());
                }
                // Записать результат в YT-таблицу
                insertYtChecksumResultRetrying(checksum, null, null, str, ChecksumStatus.STARTED);
            } else {
                throw new UnsupportedOperationException("Not supported method: " + checksum.getMethod());
            }

        } catch (RuntimeException e) {
            logger.error("Exception occurred while processing. Chunk will be marked as cancelled.", e);

            insertYtChecksumResultRetrying(checksum, null, null, e.getMessage(), ChecksumStatus.FAILED);
        }

        return true;
    }

    private void insertYtChecksumResultRetrying(ChecksumRecord checksum,
                                                @Nullable Integer ytCnt,
                                                @Nullable String ytCrc,
                                                @Nullable String ytStr,
                                                ChecksumStatus status) {
        ThreadUtils.execWithRetries((attempt) -> {
            try {
                insertYtChecksumResult(checksum, ytCnt, ytCrc, ytStr, status);
            } catch (InterruptedException ex) {
                Thread.currentThread().interrupt();
                throw new InterruptedRuntimeException(ex);
            }
        }, 5, 1_000, 2, logger);
    }

    private void insertYtChecksumResult(ChecksumRecord checksum, Integer ytCnt, String ytCrc, String ytStr,
                                        ChecksumStatus status)
            throws InterruptedException {
        var ytChecksumSchema = applyingSet.processors.stream()
                .filter(p -> p.getMainTable().getPath().endsWith(YT_CHECKSUM)).findFirst().orElseThrow()
                .getMainTable().getWriteSchema();
        CompletableFuture<Void> applyFuture;
        applyFuture = ytSupport.runTransaction(tx -> {
            FlatRow flatRow = new FlatRow(18);
            flatRow.set(0, YTree.stringNode(checksum.getTable()));
            flatRow.set(1, YTree.integerNode(checksum.getIteration()));
            flatRow.set(2, YTree.integerNode(checksum.getChunk()));
            flatRow.set(3, YTree.integerNode(checksum.getParentChunk()));
            flatRow.set(4, YTree.stringNode(checksum.getBoundaries()));
            flatRow.set(5, YTree.stringNode(checksum.getMethod().toSource()));
            flatRow.set(6, YTree.stringNode(checksum.getCrc()));
            flatRow.set(7, checksum.getCrcCnt() != null ? YTree.integerNode(checksum.getCrcCnt()) : YTree.nullNode());
            flatRow.set(8, YTree.stringNode(checksum.getStr()));
            flatRow.set(9, YTree.nullNode());
            flatRow.set(10, YTree.stringNode(ytCrc));
            flatRow.set(11, ytCnt != null ? YTree.integerNode(ytCnt) : YTree.nullNode());
            flatRow.set(12, YTree.stringNode(ytStr));
            flatRow.set(13, YTree.nullNode());
            flatRow.set(14, YTree.stringNode(checksum.getColumnsUsed()));
            flatRow.set(15, YTree.stringNode(status.toSource()));
            flatRow.set(16, YTree.stringNode(DATE_TIME_FORMATTER.format(checksum.getTs().toInstant())));
            flatRow.set(17, YTree.stringNode(getSourceColumnValue(environmentType, dbName)));

            return tx.insertRows(
                    YPath.simple(syncConfig.rootPath()).child(dbName).child("straight").child(YT_CHECKSUM).toString(),
                    ytChecksumSchema,
                    Collections.singletonList(flatRow)
            );
        });
        YtSyncUtil.joinInterruptibly(applyFuture);
    }

    private UnversionedRowset executeQuery(String query) {
        CompletableFuture<UnversionedRowset> future = ytSupport.selectRows(query);
        try {
            return future.get();
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new RuntimeException(e);
        } catch (ExecutionException e) {
            throw new RuntimeException(e);
        }
    }

    private static final String TO_STR_SEPARATOR = "\n\n";
    private static final int TO_STR_MAX_LEN = 10240;

    private String rowsToString(List<UnversionedRow> rows, List<ColumnSchema> columns, List<Integer> nullableColumns) {
        // Если чанк пуст, то результатом должен быть null
        if (rows.isEmpty()) {
            return null;
        }
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < rows.size(); i++) {
            UnversionedRow row = rows.get(i);
            sb.append(rowToString(row, columns, nullableColumns));
            if (i < rows.size() - 1) {
                sb.append(TO_STR_SEPARATOR);
            }
            if (sb.length() >= TO_STR_MAX_LEN) {
                sb.setLength(TO_STR_MAX_LEN);
                break;
            }
        }
        return sb.toString();
    }

    private String rowToString(UnversionedRow row, List<ColumnSchema> columns, List<Integer> nullableColumns) {
        List<UnversionedValue> values = row.getValues();
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < values.size(); i++) {
            UnversionedValue value = values.get(i);
            String stringValue = getStringValue(value, columns.get(i));
            // Так же с null'ами поступает mysql, когда вычисляет WS_CONCAT('#', 'sdf', null, 'fsd')
            if (stringValue != null) {
                if (i > 0) {
                    sb.append('#');
                }
                sb.append(stringValue);
            }
        }
        // Теперь аппендим nullable-колонки
        sb.append('#');
        if (!nullableColumns.isEmpty()) {
            for (Integer columnIndex : nullableColumns) {
                sb.append(values.get(columnIndex).getType() == ColumnValueType.NULL ? "1" : "0");
            }
        }
        return sb.toString();
    }

    // Возвращает индексы nullable колонок в таблице
    private List<Integer> nullableColumns(List<ColumnSchema> columns) {
        List<Integer> nullableColumns = new ArrayList<>();
        for (int i = 0; i < columns.size(); i++) {
            if (columns.get(i).isNullable()) {
                nullableColumns.add(i);
            }
        }
        return nullableColumns;
    }

    private static class CrcResult {
        final int cnt;
        final String crc;

        CrcResult(int cnt, String crc) {
            this.cnt = cnt;
            this.crc = crc;
        }
    }

    private CrcResult computeCrc(List<UnversionedRow> rows, List<ColumnSchema> columns) {
        // Индексы nullable-колонок
        List<Integer> nullableColumns = nullableColumns(columns);
        //
        int cnt = 0;
        String crc = "";
        for (UnversionedRow row : rows) {
            String rowString = rowToString(row, columns, nullableColumns);
            String s = getHexCrc32(crc + getCrc32(rowString));
            crc = StringUtils.leftPad(Long.toString(++cnt), 16, '0') + s;
        }
        return new CrcResult(cnt, crc.isEmpty() ? "0" : getRightSubstr(crc, 16));
    }

    private String getRightSubstr(String s, int right) {
        if (s.length() < right) {
            throw new IllegalArgumentException("s is too short");
        }
        return s.substring(s.length() - right);
    }

    private String getCrc32(String text) {
        CRC32 crc32 = new CRC32();
        crc32.update(text.getBytes(StandardCharsets.UTF_8));
        return Long.toString(crc32.getValue());
    }

    private String getHexCrc32(String text) {
        CRC32 crc32 = new CRC32();
        crc32.update(text.getBytes(StandardCharsets.UTF_8));
        return Long.toHexString(crc32.getValue()).toUpperCase();
    }

    static double truncate(double value, int places) {
        BigDecimal bigDecimal = new BigDecimal(value, new MathContext(15));
        return bigDecimal.setScale(places, RoundingMode.DOWN).doubleValue();
    }

    // TODO : написать юнит-тесты на это, и подробнее разобраться в возможной точности даблов на входе
    static double round(double value, int places) {
        // Округляем аналогично тому как это делает mysql для DOUBLE
        // https://dev.mysql.com/doc/refman/5.7/en/precision-math-rounding.html
        // https://en.wikipedia.org/wiki/Rounding#Round_half_to_even
        BigDecimal bigDecimal = new BigDecimal(value,
                // Если не указать явно 15 значащих цифр, то bigDecimal при округлении может дать ошибку,
                // т.к. ошибка уже есть в представлении double на входе
                // Например если создать new BigDecimal(128.865D) то мы получим
                // внутри такое представление 128865000000000009094947017729282379150390625
                // которое уже будет ближе к 128866, и при округлении до 2 позиций даст 128.87 вместо 128.86
                new MathContext(15, RoundingMode.HALF_EVEN));
        return bigDecimal.setScale(places, RoundingMode.HALF_EVEN).doubleValue();
    }

    // NB: Not thread safe class !
    public static final DecimalFormat DECIMAL_FORMAT = new DecimalFormat("0.000");

    private String getStringValue(UnversionedValue unversionedValue, ColumnSchema columnSchema) {
        switch (unversionedValue.getType()) {
            case STRING: {
                // Некоторые значения мы по факту трактуем как null'ы: и при чтении из MySQL, и при чтении из YT
                // В частности, используем опцию ZERO_DATETIME_BEHAVIOR_CONVERT_TO_NULL в DataSourceFactory
                String value = unversionedValue.stringValue();
                if (columnSchema.getDataType().equals("timestamp")) {
                    if ("1970-01-01 03:00:00".equals(value) || "0000-00-00 00:00:00".equals(value)) {
                        return null;
                    }
                } else if (columnSchema.getDataType().equals("datetime")) {
                    if ("0000-00-00 00:00:00".equals(value)) {
                        return null;
                    }
                } else if (columnSchema.getDataType().equals("date")) {
                    if ("0000-00-00".equals(value)) {
                        return null;
                    }
                } else if (columnSchema.getDataType().equals("time")) {
                    if ("00:00:00".equals(value)) {
                        return null;
                    }
                }
                return value;
            }
            case INT64:
                return Long.toString(unversionedValue.longValue());
            case UINT64: {
                // Если число не влезает в long, то восстанавливаем его беззнаковое представление
                return Long.toUnsignedString(unversionedValue.longValue());
            }
            case DOUBLE:
                // TRUNCATE(V, 3) по аналогии с тем, что делается на стороне mysql
                return DECIMAL_FORMAT.format(truncate(unversionedValue.doubleValue(), 3));
            case BOOLEAN:
                return Boolean.toString(unversionedValue.booleanValue());
            case NULL:
                return null;
            default:
                // Остальные специфичные типы игнорируем
                return null;
        }
    }

    private static class TakenOperationsInfo {
        int opsCount;
        long firstBatchTimestamp;
        long takeStartNanoTime;
    }

    private static class TableSet {
        private final List<TableProcessor> processors;

        TableSet(List<TableProcessor> processors) {
            this.processors = processors;
        }
    }
}
