package ru.yandex.direct.mysql.ytsync.synchronizator.streamer.yt;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.function.BiFunction;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.direct.env.EnvironmentType;
import ru.yandex.direct.mysql.ytsync.common.compatibility.YtSupport;
import ru.yandex.direct.mysql.ytsync.common.components.SyncStatesConfig;
import ru.yandex.direct.mysql.ytsync.common.components.SyncStatesTable;
import ru.yandex.direct.mysql.ytsync.common.components.SyncStatesTableFactory;
import ru.yandex.direct.mysql.ytsync.synchronizator.streamer.mysql.MysqlTransactionAggregator;
import ru.yandex.direct.mysql.ytsync.synchronizator.streamer.mysql.MysqlTransactionAggregatorFactory;
import ru.yandex.direct.mysql.ytsync.synchronizator.tableprocessors.TableProcessor;
import ru.yandex.direct.mysql.ytsync.synchronizator.tables.Table;
import ru.yandex.direct.mysql.ytsync.synchronizator.util.SyncConfig;
import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.inside.yt.kosher.cypress.CypressNodeType;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;

import static ru.yandex.direct.mysql.ytsync.common.util.YtSyncCommonUtil.waitAll;

@Component
@ParametersAreNonnullByDefault
public class YtTransactionAggregatorFactory implements MysqlTransactionAggregatorFactory, SyncStatesTableFactory {
    private final SyncConfig syncConfig;
    private final Yt yt;
    private final YtSupport ytSupport;
    private final Supplier<List<TableProcessor>> tableProcessorsProvider;
    private final SyncStatesConfig syncStatesConfig;
    private final Consumer<Map<Integer, Long>> timeLagConsumer;
    private final BiFunction<YPath, String, TableProcessor> yPathToTaskProcessorConverter;
    private final Function<String, List<TableProcessor>> perDbNameTableProcessorsProvider;
    private final EnvironmentType environmentType;

    private static final Logger logger = LoggerFactory.getLogger(YtTransactionAggregatorFactory.class);

    public YtTransactionAggregatorFactory(EnvironmentType environmentType,
                                          SyncConfig syncConfig, Yt yt,
                                          YtSupport ytSupport, Supplier<List<TableProcessor>> tableProcessorsProvider,
                                          SyncStatesConfig syncStatesConfig,
                                          Consumer<Map<Integer, Long>> timeLagConsumer,
                                          BiFunction<YPath, String, TableProcessor> yPathToTaskProcessorConverter,
                                          Function<String, List<TableProcessor>> perDbNameTableProcessorsProvider) {
        this.environmentType = environmentType;
        this.syncConfig = syncConfig;
        this.yt = yt;
        this.ytSupport = ytSupport;
        this.tableProcessorsProvider = tableProcessorsProvider;
        this.syncStatesConfig = syncStatesConfig;
        this.timeLagConsumer = timeLagConsumer;
        this.yPathToTaskProcessorConverter = yPathToTaskProcessorConverter;
        this.perDbNameTableProcessorsProvider = perDbNameTableProcessorsProvider;
    }

    private List<Future<?>> beginMakeTablesMounted(YPath parent, List<Table> tables,
            ExecutorService executorService) {
        Set<String> mounted = yt.cypress().list(parent, Cf.set("type", "dynamic", "tablet_state")).stream().
                filter(node -> "table".equals(node.getAttribute("type").map(YTreeNode::stringValue).orElse(null))
                        && node.getAttribute("dynamic").get().boolValue()
                        && "mounted".equals(node.getAttribute("tablet_state").get().stringValue()))
                .map(YTreeNode::stringValue)
                .collect(Collectors.toSet());

        List<Future<?>> futures = new ArrayList<>();
        for (Table table : tables) {
            String path = table.getPath();
            if (!mounted.contains(YPath.simple(path).name())) {
                futures.add(executorService.submit(() -> {
                    logger.info("Checking tablet state of table {}", table.getPath());
                    table.makeTableMounted();
                }));
            }
        }
        return futures;
    }

    public void prepareSync() {
        // Проверяем и монтируем все зарегистрированные таблицы
        // Тред пул используем не слишком большой, чтобы не слишком нагружать YT (у нас уже около десятка инстансов)
        ExecutorService executorService = Executors.newFixedThreadPool(10);
        try {
            List<Table> tables = tableProcessorsProvider.get().stream()
                    .map(TableProcessor::getMainTable)
                    .collect(Collectors.toList());
            logger.info("Starting checking and mounting/unfreezing {} tables..", tables.size());
            StreamEx.of(tables)
                    .groupingBy(table -> YPath.simple(table.getPath()).parent())
                    .forEach((parent, tableList) -> waitAll(beginMakeTablesMounted(parent, tableList, executorService)));
            logger.info("Tables checking finished");
        } finally {
            // Без остановки этого пула программа не сможет завершиться
            logger.info("Stopping prepareSync executorService..");
            executorService.shutdownNow();
            logger.info("Stopped prepareSync executorService");
        }

        for (String dbName : syncConfig.getDbNames()) {
            YPath lockPath = getLockPath(dbName);
            if (!yt.cypress().exists(lockPath)) {
                yt.cypress().create(lockPath, CypressNodeType.BOOLEAN, true, true);
            }
        }
    }

    @Override
    public SyncStatesTable getSyncStatesTable() {
        return new SyncStatesTable(
                ytSupport,
                syncStatesConfig.syncStatesPath(),
                syncStatesConfig.syncTableMedium(),
                timeLagConsumer
        );
    }

    private YPath getLockPath(String dbName) {
        return YPath.simple(syncConfig.syncLocksPrefix() + dbName);
    }

    @Override
    public MysqlTransactionAggregator getAggregator(String dbName) {
        YtTransactionAggregator ytTransactionAggregator =
                new YtTransactionAggregator(environmentType, dbName, getLockPath(dbName), ytSupport, yt, syncConfig,
                        getSyncStatesTable(), perDbNameTableProcessorsProvider, yPathToTaskProcessorConverter);
        ytTransactionAggregator.start();
        return ytTransactionAggregator;
    }
}
