package ru.yandex.direct.mysql.ytsync.synchronizator.tables;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.mysql.ytsync.common.compatibility.YtSupport;
import ru.yandex.direct.mysql.ytsync.common.row.FlatRow;
import ru.yandex.direct.mysql.ytsync.common.tables.TableWriteSnapshot;
import ru.yandex.direct.mysql.ytsync.synchronizator.util.YtSyncUtil;
import ru.yandex.yt.ytclient.tables.TableSchema;

/**
 * Базовый класс для таблиц, у которых есть индекс для основного ключа mysql
 */
public class TableBaseWithIndex extends TableBase {
    private static final Logger logger = LoggerFactory.getLogger(TableBaseWithIndex.class);

    private final Table indexTable;

    // Для конвертации текущей строки в строку индекса
    private final int[] indexRowFromTableRow;

    // Для конвертации индексной строки в текущий ключ
    private final int[] tableKeyFromIndexRow;

    // Конвертирует ключ индекса в соответствующий ключ текущей строки
    private final Map<FlatRow, FlatRow> knownIndexKeys = new HashMap<>();

    // Конвертирует ключ индекса в список операций, для которых не хватает ключевых колонок
    private final Map<FlatRow, List<IncompleteOp>> incompleteOps = new HashMap<>();

    public TableBaseWithIndex(
            YtSupport yt,
            String path,
            TableSchema schema,
            Table indexTable) {
        super(yt, path, schema);
        this.indexTable = indexTable;

        this.indexRowFromTableRow = new int[indexTable.getWriteSchema().getColumnsCount()];
        for (int index = 0; index < indexTable.getWriteSchema().getColumnsCount(); index++) {
            String column = indexTable.getWriteSchema().getColumnName(index);
            int currentIndex = getWriteSchema().findColumn(column);
            if (currentIndex == -1) {
                throw new IllegalArgumentException("Cannot find column " + column + " in table " + path);
            }
            indexRowFromTableRow[index] = currentIndex;
        }

        this.tableKeyFromIndexRow = new int[getWriteSchema().getKeyColumnsCount()];
        for (int currentIndex = 0; currentIndex < getWriteSchema().getKeyColumnsCount(); currentIndex++) {
            String column = getWriteSchema().getColumnName(currentIndex);
            int index = indexTable.getWriteSchema().findColumn(column);
            if (index == -1) {
                throw new IllegalArgumentException(
                        "Cannot find column " + column + " in table " + indexTable.getPath());
            }
            tableKeyFromIndexRow[currentIndex] = index;
        }
    }

    /**
     * Возвращает индексную таблицу
     */
    public Table getIndexTable() {
        return indexTable;
    }

    /**
     * Доступ к известным на данных момент ключам индекса, для тестов
     */
    Map<FlatRow, FlatRow> getKnownIndexKeys() {
        return knownIndexKeys;
    }

    /**
     * Доступ к спискам операций, для которых не хватает данных, для тестов
     */
    Map<FlatRow, List<IncompleteOp>> getIncompleteOps() {
        return incompleteOps;
    }

    private static void fillMissing(FlatRow src, FlatRow dst) {
        int size = Math.min(src.size(), dst.size());
        for (int index = 0; index < size; index++) {
            if (dst.get(index) == null) {
                dst.set(index, src.get(index));
            }
        }
    }

    private FlatRow makeIndexKeyFromTableRow(FlatRow tableRow) {
        FlatRow indexKey = new FlatRow(indexTable.getWriteSchema().getKeyColumnsCount());
        for (int index = 0; index < indexKey.size(); index++) {
            indexKey.set(index, tableRow.get(indexRowFromTableRow[index]));
        }
        return indexKey;
    }

    private FlatRow makeIndexRowFromTableRow(FlatRow tableRow) {
        FlatRow indexRow = new FlatRow(indexTable.getWriteSchema().getColumnsCount());
        for (int index = 0; index < indexRow.size(); index++) {
            indexRow.set(index, tableRow.get(indexRowFromTableRow[index]));
        }
        return indexRow;
    }

    private FlatRow makeTableKeyFromIndexRow(FlatRow indexRow) {
        FlatRow tableKey = new FlatRow(getWriteSchema().getKeyColumnsCount());
        for (int index = 0; index < tableKey.size(); index++) {
            tableKey.set(index, indexRow.get(tableKeyFromIndexRow[index]));
        }
        return tableKey;
    }

    private void addIncompleteOp(FlatRow indexKey, IncompleteOp op) {
        List<IncompleteOp> ops = incompleteOps.computeIfAbsent(indexKey, k -> new ArrayList<>());
        ops.add(op);
    }

    private void addKnownIndexKey(FlatRow indexKey, FlatRow tableKey) {
        knownIndexKeys.put(indexKey, tableKey);
        List<IncompleteOp> ops = incompleteOps.remove(indexKey);
        if (ops != null) {
            for (IncompleteOp op : ops) {
                op.reapply();
            }
        }
    }

    @Override
    public void makeTableMounted() {
        indexTable.makeTableMounted();
        super.makeTableMounted();
    }

    @Override
    public void clear() {
        super.clear();
        indexTable.clear();
        knownIndexKeys.clear();
        incompleteOps.clear();
    }

    @Override
    public int addDelete(FlatRow row) {
        FlatRow indexKey = makeIndexKeyFromTableRow(row);
        if (indexKey.hasMissingValues()) {
            // На реальных таблицах мы всегда должны иметь возможность получить ключ индекса
            logger.error("Attempt to delete incomplete row: {}", row);
            return 0;
        }

        FlatRow tableKey = row.first(getWriteSchema().getKeyColumnsCount());
        if (tableKey.hasMissingValues()) {
            // В переданной строке есть неизвестные нам ключевые колонки
            tableKey = knownIndexKeys.get(indexKey);
            if (tableKey == null) {
                // Откладываем операцию до появления полного ключа
                addIncompleteOp(indexKey, new IncompleteDelete(row));
                return 2;
            }
            fillMissing(tableKey, row);
        } else {
            addKnownIndexKey(indexKey, tableKey);
        }
        return indexTable.addDelete(indexKey) + super.addDelete(row);
    }

    @Override
    public int addInsert(FlatRow row) {
        FlatRow indexRow = makeIndexRowFromTableRow(row);
        FlatRow indexKey = indexRow.first(indexTable.getWriteSchema().getKeyColumnsCount());
        if (indexKey.hasMissingValues()) {
            logger.error("Attempt to insert incomplete row: {}", row);
            return 0;
        }

        FlatRow tableKey = row.first(getWriteSchema().getKeyColumnsCount());
        if (tableKey.hasMissingValues()) {
            logger.error("Attempt to insert row without a complete key: {}", row);
            return 0;
        }

        addKnownIndexKey(indexKey, tableKey);
        return indexTable.addInsert(indexRow) + super.addInsert(row);
    }

    @Override
    public int addUpdate(FlatRow before, FlatRow after) {
        if (before == null) {
            before = after.copy();
            for (int index = getWriteSchema().getKeyColumnsCount(); index < getWriteSchema().getColumnsCount(); index++) {
                before.set(index, null);
            }
        } else {
            for (int index = 0; index < getWriteSchema().getColumnsCount(); index++) {
                if (after.get(index) == null) {
                    after.set(index, before.get(index));
                }
            }
        }

        FlatRow indexRowBefore = makeIndexRowFromTableRow(before);
        FlatRow indexRowAfter = makeIndexRowFromTableRow(after);
        FlatRow indexKeyBefore = indexRowBefore.first(indexTable.getWriteSchema().getKeyColumnsCount());
        FlatRow indexKeyAfter = indexRowAfter.first(indexTable.getWriteSchema().getKeyColumnsCount());
        if (indexKeyBefore.hasMissingValues() || indexKeyAfter.hasMissingValues()) {
            logger.error("Attempt to update incomplete row: {} -> {}", before, after);
            return 0;
        }

        FlatRow tableKeyBefore = before.first(getWriteSchema().getKeyColumnsCount());
        if (tableKeyBefore.hasMissingValues()) {
            // В операции обновления нет полного исходного ключа
            tableKeyBefore = knownIndexKeys.get(indexKeyBefore);
            if (tableKeyBefore == null) {
                // Откладываем операцию пока не узнаем ключ
                addIncompleteOp(indexKeyBefore, new IncompleteUpdate(before, after));
                return 1;
            }
            fillMissing(tableKeyBefore, before);
            fillMissing(tableKeyBefore, after);
            indexRowBefore = makeIndexRowFromTableRow(before);
            indexRowAfter = makeIndexRowFromTableRow(after);
        } else {
            addKnownIndexKey(indexKeyBefore, tableKeyBefore);
        }

        FlatRow tableKeyAfter = after.first(getWriteSchema().getKeyColumnsCount());
        if (tableKeyAfter.hasMissingValues()) {
            // На более раннем этапе мы заполнили все недостающие колонки в before
            // Соответственно недостающие колонки в after также должны были заполниться
            logger.error("Unexpected incomplete after key: {} -> {}", before, after);
            return 0;
        }

        int count = 0;
        if (!indexKeyAfter.equals(indexKeyBefore)) {
            // В индексе меняется primary key, делаем DELETE + INSERT
            count += indexTable.addDelete(indexKeyBefore);
            count += indexTable.addInsert(indexRowAfter);
        } else if (!indexRowAfter.equals(indexRowBefore)) {
            // В индексе меняется значение, делаем простой UPDATE
            count += indexTable.addUpdate(indexRowBefore, indexRowAfter);
        }
        count += super.addUpdate(before, after);
        // После выполнения операции нам может стать известно новое значение ключа
        addKnownIndexKey(indexKeyAfter, tableKeyAfter);
        return count;
    }

    @Override
    public CompletableFuture<List<TableWriteSnapshot>> prepare(YtSupport.Transaction tx) {
        if (!incompleteOps.isEmpty()) {
            // Грузим все ключи из incompleteOps и перезапускаемся
            final long tstart = System.nanoTime();
            final List<FlatRow> neededKeys = new ArrayList<>(incompleteOps.keySet());
            return indexTable.lookupRows(tx, neededKeys).thenComposeAsync(indexRows -> {
                long tend = System.nanoTime();
                logger.info("Looking up all {} incomplete keys took {}ms", neededKeys.size(),
                        TimeUnit.NANOSECONDS.toMillis(tend - tstart));
                for (FlatRow indexRow : indexRows) {
                    FlatRow indexKey = indexRow.first(indexTable.getWriteSchema().getKeyColumnsCount());
                    if (knownIndexKeys.get(indexKey) == null) {
                        FlatRow tableKey = makeTableKeyFromIndexRow(indexRow);
                        addKnownIndexKey(indexKey, tableKey);
                    }
                }
                int unhandled = 0;
                for (FlatRow indexKey : neededKeys) {
                    List<IncompleteOp> ops = incompleteOps.remove(indexKey);
                    if (ops != null) {
                        // Если в incompleteOps остались какие-то операции, то мы уже ничего не можем с ними сделать
                        unhandled += ops.size();
                    }
                }
                if (unhandled > 0) {
                    logger.error("Could not handle {} incomplete operations", unhandled);
                }
                return prepare(tx);
            }, tx.executor());
        }
        CompletableFuture<List<TableWriteSnapshot>> tableSnapshot = super.prepare(tx);
        CompletableFuture<List<TableWriteSnapshot>> indexSnapshot = indexTable.prepare(tx);
        return YtSyncUtil.collectAllOrdered(Arrays.asList(tableSnapshot, indexSnapshot));
    }

    /**
     * Операция, для выполнения которой не хватает данных
     */
    private interface IncompleteOp {
        void reapply();
    }

    private class IncompleteDelete implements IncompleteOp {
        private final FlatRow row;

        public IncompleteDelete(FlatRow row) {
            this.row = row;
        }


        @Override
        public void reapply() {
            addDelete(row);
        }
    }

    private class IncompleteUpdate implements IncompleteOp {
        private final FlatRow before;
        private final FlatRow after;

        public IncompleteUpdate(FlatRow before, FlatRow after) {
            this.before = before;
            this.after = after;
        }

        @Override
        public void reapply() {
            addUpdate(before, after);
        }
    }
}
