package ru.yandex.direct.mysql.ytsync.synchronizator.tables;

import java.util.AbstractList;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.RandomAccess;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;

import com.google.common.collect.Collections2;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.mysql.ytsync.common.compatibility.YtSupport;
import ru.yandex.direct.mysql.ytsync.common.row.FlatRow;
import ru.yandex.direct.mysql.ytsync.common.row.FlatRowView;
import ru.yandex.direct.mysql.ytsync.common.tables.TableWriteSnapshot;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTree;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;
import ru.yandex.yt.ytclient.tables.TableSchema;

/**
 * Буфер для записи данных в динамическую yt таблицу
 * <p>
 * Запрещается одновременное использование из нескольких потоков
 */
public class TableWriteBuffer implements TableWriteSink, TableReadSource {
    private static final Logger logger = LoggerFactory.getLogger(TableWriteBuffer.class);

    private final String path;
    private final TableSchema writeSchema;
    private final TableSchema lookupSchema;

    private final Set<FlatRow> deletes = new HashSet<>();
    private final Map<FlatRow, FlatRow> inserts = new HashMap<>();
    private final Map<FlatRow, FlatRow> updates = new HashMap<>();
    private final Map<FlatRow, FlatRow> remapping = new HashMap<>();

    private volatile boolean consistencyViolationFound = false;

    public TableWriteBuffer(String path, TableSchema schema) {
        this.path = Objects.requireNonNull(path);
        this.writeSchema = schema.toWrite();
        this.lookupSchema = writeSchema.toLookup();
    }

    @Override
    public String getPath() {
        return path;
    }

    @Override
    public TableSchema getWriteSchema() {
        return writeSchema;
    }

    @Override
    public TableSchema getLookupSchema() {
        return lookupSchema;
    }

    /**
     * Запланированные на удаление ключи, для тестов
     */
    Collection<FlatRowView> getDeletes() {
        return Collections.unmodifiableSet(deletes);
    }

    /**
     * Запланированные на вставку строки, для тестов
     */
    Collection<FlatRowView> getInserts() {
        return Collections2.transform(inserts.entrySet(), entry -> {
            Objects.requireNonNull(entry);
            return new JoinedKeyValue(entry.getKey(), entry.getValue());
        });
    }

    /**
     * Запланированные на обновление строки, для тестов
     */
    Collection<FlatRowView> getUpdates() {
        return Collections2.transform(updates.entrySet(), entry -> {
            Objects.requireNonNull(entry);
            return new JoinedKeyValue(entry.getKey(), entry.getValue());
        });
    }

    /**
     * Запланированные на чтение пары ключей, для тестов
     */
    Collection<FlatRowView> getRemapping() {
        return Collections2.transform(remapping.entrySet(), entry -> {
            Objects.requireNonNull(entry);
            return new JoinedKeyValue(entry.getKey(), entry.getValue());
        });
    }

    /**
     * Очищает текущий буфер
     */
    @Override
    public void clear() {
        deletes.clear();
        inserts.clear();
        updates.clear();
        remapping.clear();
        consistencyViolationFound = false;
    }

    private boolean removeRawDelete(FlatRow key) {
        return deletes.remove(key);
    }

    private boolean removeRawInsert(FlatRow key) {
        return inserts.remove(key) != null;
    }

    private boolean removeRawUpdate(FlatRow key) {
        if (updates.remove(key) != null) {
            remapping.remove(key);
            return true;
        } else {
            return false;
        }
    }

    private boolean addRawDelete(FlatRow key) {
        return deletes.add(key);
    }

    private boolean addRawInsert(FlatRow key, FlatRow value) {
        return inserts.put(key, value) == null;
    }

    private boolean addRawUpdate(FlatRow key, FlatRow value) {
        return updates.put(key, value) == null;
    }

    private boolean addRawUpdate(FlatRow key, FlatRow value, FlatRow oldKey) {
        remapping.put(key, oldKey);
        return addRawUpdate(key, value);
    }

    @Override
    public int addDelete(FlatRow row) {
        if (row.size() != writeSchema.getKeyColumnsCount() && row.size() != writeSchema.getColumnsCount()) {
            throw new IllegalArgumentException("Invalid number of columns");
        }

        FlatRow key = row.first(writeSchema.getKeyColumnsCount());
        key.fillMissing(YTree.builder().entity().build());

        int count = 0;
        if (removeRawUpdate(key) || removeRawInsert(key)) {
            --count;
        }
        if (addRawDelete(key)) {
            ++count;
        }
        return count;
    }

    @Override
    public int addInsert(FlatRow row) {
        if (row.size() != writeSchema.getColumnsCount()) {
            throw new IllegalArgumentException("Invalid number of columns");
        }

        FlatRow key = row.first(writeSchema.getKeyColumnsCount());
        FlatRow value = row.copy(writeSchema.getKeyColumnsCount(), writeSchema.getColumnsCount());
        key.fillMissing(YTree.builder().entity().build());
        value.fillMissing(YTree.builder().entity().build());

        int count = 0;
        if (removeRawUpdate(key) || removeRawDelete(key)) {
            --count;
        }
        if (addRawInsert(key, value)) {
            ++count;
        }
        return count;
    }

    @Override
    public int addUpdate(FlatRow row) {
        if (row.size() != writeSchema.getColumnsCount()) {
            throw new IllegalArgumentException("Invalid number of columns");
        }

        return addUpdate(null, row);
    }

    @Override
    public int addUpdate(FlatRow before, FlatRow after) {
        // Для упрощения вышестоящего кода before может быть равен null
        if (before != null && before.size() != writeSchema.getKeyColumnsCount() &&
                before.size() != writeSchema.getColumnsCount()) {
            throw new IllegalArgumentException("Invalid number of before columns");
        }
        if (after.size() != writeSchema.getColumnsCount()) {
            throw new IllegalArgumentException("Invalid number of after columns");
        }

        FlatRow oldKey = (before != null ? before : after).first(writeSchema.getKeyColumnsCount());
        oldKey.fillMissing(YTree.builder().entity().build());

        FlatRow newKey = null;
        if (before != null) {
            // Проверяем нет ли у нас изменений в колонках ключа
            for (int i = 0; i < writeSchema.getKeyColumnsCount(); i++) {
                YTreeNode oldValue = oldKey.get(i);
                YTreeNode newValue = after.get(i);
                if (newValue != null && !newValue.equals(oldValue)) {
                    if (newKey == null) {
                        newKey = oldKey.copy();
                    }
                    newKey.set(i, newValue);
                }
            }
        }

        List<YTreeNode> valueUpdates = after.subList(writeSchema.getKeyColumnsCount(), writeSchema.getColumnsCount());
        if (before != null && before.size() == writeSchema.getColumnsCount()) {
            // Убираем лишние update'ы, чтобы пользовательскому коду не приходилось это делать
            valueUpdates = new RemoveRedundantUpdates(
                    before.subList(writeSchema.getKeyColumnsCount(), writeSchema.getColumnsCount()),
                    valueUpdates);
        }

        if (deletes.contains(oldKey)) {
            // Попытка изменить ранее удалённую строку
            logger.warn("Found consistency violation on {} (key {}): UPDATE after DELETE", path, oldKey);
            consistencyViolationFound = true;
            return 0;
        }

        int count = 0;
        FlatRow previousValue;
        if (newKey != null) {
            // Мы точно знаем, что newKey != oldKey и произошло изменение ключа
            // Для начала убираем всё, что было про newKey, мы его перезаписываем
            if (removeRawDelete(newKey) || removeRawUpdate(newKey) || removeRawInsert(newKey)) {
                --count;
            }
            if (addRawDelete(oldKey)) {
                // should always be true
                ++count;
            }
            if ((previousValue = inserts.remove(oldKey)) != null) {
                --count;
                // У нас был INSERT на старый ключ, заменяем на DELETE + INSERT с новым ключом
                previousValue.update(valueUpdates);
                if (addRawInsert(newKey, previousValue)) {
                    ++count;
                }
            } else if ((previousValue = updates.remove(oldKey)) != null) {
                --count;
                // У нас был UPDATE на старый ключ, заменяем на DELETE + UPDATE с ремаппингом, если нужно
                FlatRow previousSourceKey = remapping.remove(oldKey);
                if (previousSourceKey == null) {
                    previousSourceKey = oldKey;
                }
                previousValue.update(valueUpdates);
                if (addRawUpdate(newKey, previousValue, previousSourceKey)) {
                    ++count;
                }
            } else {
                if (addRawUpdate(newKey, FlatRow.of(valueUpdates), oldKey)) {
                    ++count;
                }
            }
        } else if ((previousValue = inserts.get(oldKey)) != null) {
            previousValue.update(valueUpdates);
        } else if ((previousValue = updates.get(oldKey)) != null) {
            previousValue.update(valueUpdates);
        } else {
            FlatRow updates = FlatRow.of(valueUpdates);
            if (updates.hasPresentValues() && addRawUpdate(oldKey, updates)) {
                ++count;
            }
        }
        return count;
    }

    private CompletableFuture<Void> handleRemapping(YtSupport.Transaction tx) {
        Set<FlatRow> neededKeys = new HashSet<>();
        Set<FlatRow> removedKeys = new HashSet<>();
        for (Map.Entry<FlatRow, FlatRow> entry : remapping.entrySet()) {
            FlatRow targetKey = entry.getKey();
            FlatRow sourceKey = entry.getValue();
            FlatRow pendingUpdate = updates.get(targetKey);
            if (pendingUpdate != null) {
                if (pendingUpdate.hasMissingValues()) {
                    // Нам нужно прочитать строку, чтобы сделать INSERT из UPDATE
                    neededKeys.add(sourceKey);
                } else {
                    // У нас уже есть все необходимые для INSERT данные
                    updates.remove(targetKey);
                    inserts.put(targetKey, pendingUpdate);
                    removedKeys.add(targetKey);
                }
            }
        }
        remapping.keySet().removeAll(removedKeys);
        if (neededKeys.isEmpty()) {
            return CompletableFuture.completedFuture(null);
        }
        logger.info("Fetching {} rows for remapped keys", neededKeys.size());
        long tstart = System.nanoTime();
        return lookupRows(tx, new ArrayList<>(neededKeys), writeSchema).thenAcceptAsync(rows -> {
            long tend = System.nanoTime();
            logger.info("Fetched {} original rows for remapped keys in {}ms", rows.size(),
                    TimeUnit.NANOSECONDS.toMillis(tend - tstart));
            Map<FlatRow, FlatRow> fetchedMap = new HashMap<>();
            for (FlatRow row : rows) {
                FlatRow key = row.first(writeSchema.getKeyColumnsCount());
                FlatRow value = row.copy(writeSchema.getKeyColumnsCount(), writeSchema.getColumnsCount());
                fetchedMap.put(key, value);
            }
            for (Map.Entry<FlatRow, FlatRow> entry : remapping.entrySet()) {
                FlatRow targetKey = entry.getKey();
                FlatRow sourceKey = entry.getValue();
                FlatRow pendingUpdate = updates.remove(targetKey);
                if (pendingUpdate != null) {
                    FlatRow sourceValues = fetchedMap.get(sourceKey);
                    if (sourceValues != null) {
                        pendingUpdate.updateWeakly(sourceValues);
                        inserts.put(targetKey, pendingUpdate);
                    } else {
                        // изменения в ключе могли произойти в промежутке
                        // от момента фиксирования GTID и до окончания наливки таблицы
                        logger.warn("Seems like key {} is already remapped to {}, ignoring", sourceKey, targetKey);
                        // в зачитанный буфер также могли попасть изменения в данных
                        // с GTID старше текущего GTID строки в таблице YT
                        if (pendingUpdate.hasPresentValues()) {
                            updates.put(targetKey, pendingUpdate);
                        }
                    }
                }
            }
            remapping.clear();
        }, tx.executor());
    }

    @Override
    public CompletableFuture<List<TableWriteSnapshot>> prepare(YtSupport.Transaction tx) {
        return handleRemapping(tx).thenApply(ignored -> {
            List<FlatRowView> insertedList;
            List<FlatRowView> updatedList;
            List<FlatRowView> deletedList;
            insertedList = new ArrayList<>(inserts.size());
            for (Map.Entry<FlatRow, FlatRow> entry : inserts.entrySet()) {
                insertedList.add(new JoinedKeyValue(entry.getKey(), entry.getValue()));
            }
            updatedList = new ArrayList<>(updates.size());
            for (Map.Entry<FlatRow, FlatRow> entry : updates.entrySet()) {
                if (entry.getValue().hasPresentValues()) {
                    updatedList.add(new JoinedKeyValue(entry.getKey(), entry.getValue()));
                }
            }
            deletedList = new ArrayList<>(deletes);
            if (insertedList.isEmpty() && updatedList.isEmpty() && deletedList.isEmpty()) {
                return Collections.emptyList();
            }
            return Collections
                    .singletonList(new TableWriteSnapshot(path, writeSchema, insertedList, updatedList, deletedList));
        });
    }

    /**
     * Представляет отдельные ключ и значение в виде единого прозрачного списка
     */
    private static class JoinedKeyValue extends FlatRowView {
        private final FlatRowView key;
        private final FlatRowView value;

        public JoinedKeyValue(FlatRowView key, FlatRowView value) {
            this.key = key;
            this.value = value;
        }

        @Override
        public YTreeNode get(int index) {
            if (index < key.size()) {
                return key.get(index);
            } else {
                return value.get(index - key.size());
            }
        }

        @Override
        public int size() {
            return key.size() + value.size();
        }
    }

    /**
     * Убирает лишние не-null значения в апдейтах
     */
    private static class RemoveRedundantUpdates extends AbstractList<YTreeNode> implements RandomAccess {
        private final List<YTreeNode> before;
        private final List<YTreeNode> after;

        public RemoveRedundantUpdates(List<YTreeNode> before, List<YTreeNode> after) {
            this.before = before;
            this.after = after;
        }

        @Override
        public YTreeNode get(int index) {
            YTreeNode value = after.get(index);
            if (value != null && value.equals(before.get(index))) {
                // Значение колонок в before и after совпадают
                value = null;
            }
            return value;
        }

        @Override
        public int size() {
            return after.size();
        }
    }

    public boolean isConsistencyViolationFound() {
        return consistencyViolationFound;
    }
}
